/*
 *****    homebrew-radios.net   *****

 *****     Project TxrAVR       *****

 taUsart.c     program file

 Serial communication module


  USART3 - receive - input from encoders8 board
                    with Atmega2560 and upto eight rotary encoders


*/


#include "taGlobal.h"
#include "TrxAVR.h"
#include "taStarControl.h"
#include "taCharLCD.h"
#include "taUsart.h"
#include "taEATFT.h"

//UART3 baud rate
#define BAUD3 38400
#define MYUBRR_3 (Xtal/16/BAUD3-1)

#define BufRx3Size 91  // MUST BE 6n + 1 so that SkipRx3 sets on full buffer
                        // with a complete 9 char sequence (then waits for the next '$' 

static char BufRx3[BufRx3Size];
static uint8_t FirstRx3;
static uint8_t NextRx3;
static uint8_t SkipRx3;
static uint8_t LastDollar;

char StringRx3[12];
char* psrx3;
char StringRx0[12];
char* psrx0;

uint8_t uaRx3BufCount();
uint8_t uaRx3Empty();
uint8_t uaRx3Full();
void uaIntEnableRx3(unsigned char tf);
uint8_t uaReadCharRx3(char* pc);
void uaResetRx3();
uint8_t uaNumeric(char* ps, uint8_t first,uint8_t count);


void uaSerialInit()
{
	// Rx3 init
	psrx3 = StringRx3;
	UBRR3H = (unsigned char)(MYUBRR_3 >> 8);  // set baud rate
	UBRR3L = (unsigned char)MYUBRR_3;
	UCSR3C = 0x06;  //   asynchronous, no parity, 1 stop bit, 8 data bits 
	UCSR3B = 0x10;   /// set bit 4 to enable reception
	uaResetRx3();
	return;
}

///////////    Rx3  - Encoders 8 input

void uaResetRx3()
{
	SkipRx3 = 0;
	FirstRx3 = 0;
	NextRx3 = 0;
	uaIntEnableRx3(1); // enable Rx interrupt 
	return;
}


ISR(USART3_RX_vect)   // interrupt on character receieved on USART3 
{
	char c = 0;
	uaReadCharRx3(&c);
	if (c == '$')
	{
		if (SkipRx3 == 1) NextRx3 = LastDollar;
		SkipRx3 = 0;
		LastDollar = NextRx3;
		WakeupDisplay = 1;
	}
	else
	{
		if (uaRx3Empty()) return;
	}
	if (uaRx3Full() == 1) SkipRx3 = 1;
	if (SkipRx3 > 0) return;
	BufRx3[NextRx3] = c;
	NextRx3 += 1;
	if (NextRx3 >= BufRx3Size) NextRx3 = 0;
	return;
}


uint8_t uaRx3Empty()
{
	if (FirstRx3 == NextRx3) {return 1;}
	return 0;
}


uint8_t uaRx3Full()
{
	uint8_t n = NextRx3;
	if (n < FirstRx3 ) {n += BufRx3Size;}
	if ( (n - FirstRx3) >= (BufRx3Size - 1) ) return 1; // buffer full
	return 0;   // buffer not full
}


uint8_t uaRx3BufCount()
{
	if (NextRx3 >= FirstRx3)
	{return (NextRx3 - FirstRx3);}
	else
	{return (NextRx3 + BufRx3Size - FirstRx3);}
}


void uaIntEnableRx3(unsigned char tf)
{
	if (tf == 0) UCSR3B &= ~(1 << 7);  // disable RXCIE3 (bit 7)
	else UCSR3B |= (1 << 7);   // enable RXCIE3 (bit 7)
	return;
}


uint8_t uaReadCharRx3(char* pc)
{
	int timeout = 0;
	while (!(UCSR3A & 0x80))
	{
		timeout += 1;
		_delay_us(10);
		if (++timeout > 1000) {return 0;}
	}
	*pc = UDR3;
	return 1;
}


uint8_t uaNumeric(char* ps, uint8_t first,uint8_t count)
{
	uint8_t i;
	for (i = first; i < (first + count); i++)
	{
		if ( (ps[i] < 0x30) || (ps[i] > 0x39) ) {return 0;}
	}
	return 1;
}
	   

//  returns 0 if valid encoder message (buffer has 6 or more chars) 
//          1 if no encoder messaage, 
//          2 for no $ at first position (does a reset)
//          3 for parsing error
uint8_t uaCheckEncoders8(uint8_t* nenc, uint8_t* delta, uint8_t* updown, uint8_t keepbuffer)
{
	uint8_t i, j, error, cSREG;

	if (zt) {
		etReadBuf();
		cSREG = SREG;
		cli();
		if (etTFT_ENC8BufferDataAvailable() >= 3) {
			i = TFT_ENC8_Tail;
			*nenc = TFT_ENC8[i];
			i = (i + 1) & TFT_ENC8_BUFFER_MASK;
			*updown = TFT_ENC8[i];
			i = (i + 1) & TFT_ENC8_BUFFER_MASK;
			*delta = TFT_ENC8[i];
			if (keepbuffer == 0) {
				TFT_ENC8_Tail = (TFT_ENC8_Tail + 3) & TFT_ENC8_BUFFER_MASK;
			}
			SREG = cSREG;
			scWakeupDisplay();
			return 0;
		}
		SREG = cSREG;
	}

	if (uaRx3BufCount() < 6) return 1;  // no change
	if (BufRx3[FirstRx3] != '$') {
		uaResetRx3();
		return 2;   // error - first not $ - reset done
	}
	for (i = 0; i <= 5; i++) {
		j = i + FirstRx3;
		if (j >= BufRx3Size) {j -= BufRx3Size;}
		StringRx3[i] = BufRx3[j];
	}
	if (keepbuffer == 0) {
		// move circular buffer on by 6
		FirstRx3  += 6;
		if (FirstRx3 >= BufRx3Size) {FirstRx3 = FirstRx3 - BufRx3Size;}
	}
// check and parse
	error = 0;
	if (StringRx3[5] != 13) error = 1;
	StringRx3[5] = 0;
	if (!uaNumeric(psrx3, 1, 1)) error = 1;
	if (!uaNumeric(psrx3, 3, 2)) error = 1;
	*updown = 2;
	if (StringRx3[2] == '+') *updown = 1;
	if (StringRx3[2] == '-') *updown = 0;
	if (*updown == 2) error = 1;
	if (error > 0) return 3;    // parsing error 
	*nenc = StringRx3[1] - 0x30;
	*delta = 10 * (StringRx3[3] - 0x30) + StringRx3[4] - 0x30;
	scWakeupDisplay();
	return 0;
}


int uaGetNext()
{
	return NextRx3;
}
