/*
 *****    homebrew-radios.net   *****

 *****     Project TxrAVR       *****


 taTWI.c    program file
 
 TWI (I2C) bus driver program file

*/

#define _UTIL_DELAY_H_ 1

#include "taGlobal.h"
#include "taTWI.h"
#include "ta24LC512.h"
#include <compat/twi.h>
#include <util/delay.h>


#define MAX_TRIES_EEPROM 250
#define MAX_TRIES_TFT 10
#define MAX_TRIES_I2C 10
#define MAX_TRIES_RTC 20



int xeeprom_read_byte(uint16_t eeaddr, char *buf)
{
	uint8_t n;

	n = 0;
	TWBR = 3;   // see datasheet p 24.4.3 - gives 400kHz with 16Mhz clock
	TWSR = 1; 

	do {
		if (n++ >= MAX_TRIES_EEPROM) return 0;
		
		do {

			// send start cond.
			TWCR = (1 << TWINT) | (1 << TWSTA) | (1 << TWEN);
			while (!(TWCR & (1 << TWINT)));
			if (TW_STATUS == TW_MT_ARB_LOST) continue;
			if ( (TW_STATUS != TW_REP_START) && (TW_STATUS != TW_START)) { return 0;}
	
			// send 0xa0
			// 0xa0 = 1010 000 0
			// 4 bits:   <a..device-indentifier>
			// 3 bits:   <device-address set with chip pins>
			// last bit: <0..write>
			TWDR = 0xa0;
			TWCR = (1 << TWINT) | (1 << TWEN);
			while (!(TWCR & (1 << TWINT)));
			if (TW_STATUS == TW_MT_SLA_NACK) break;
			if (TW_STATUS == TW_MT_ARB_LOST) continue;
			if (TW_STATUS != TW_MT_SLA_ACK) {TWCR = (1 << TWINT) | (1 << TWEN) | (1 << TWSTO); return 0;}

			// send high 8 bits of eeaddr 
			TWDR = eeaddr / 256;
			TWCR = (1 << TWINT) | (1 << TWEN);
			while (!(TWCR & (1 << TWINT)));
			if (TW_STATUS == TW_MT_DATA_NACK) {TWCR = (1 << TWINT) | (1 << TWEN) | (1 << TWSTO); break;}
			if (TW_STATUS == TW_MT_ARB_LOST) continue;
			if (TW_STATUS != TW_MT_DATA_ACK) {TWCR = (1 << TWINT) | (1 << TWEN) | (1 << TWSTO); return 0;}

			// send low 8 bits of eeaddr
			TWDR = eeaddr % 256;
			TWCR = (1 << TWINT) | (1 << TWEN);
			while (!(TWCR & (1 << TWINT)));
			if (TW_STATUS == TW_MT_DATA_NACK) {TWCR = (1 << TWINT) | (1 << TWEN) | (1 << TWSTO); break;}
			if (TW_STATUS == TW_MT_ARB_LOST) continue;
			if (TW_STATUS != TW_MT_DATA_ACK) {TWCR = (1 << TWINT) | (1 << TWEN) | (1 << TWSTO); return 0;}

			// send start cond.
			TWCR = (1 << TWINT) | (1 << TWSTA) | (1 << TWEN);
			while (!(TWCR & (1 << TWINT)));
			if (TW_STATUS == TW_MT_ARB_LOST) continue;
			if ( (TW_STATUS != TW_REP_START) && (TW_STATUS != TW_START)) {return 0;}

			// send 0xa1
			// 0xa1 = 1010 000 1
			// 4 bits:   <a..device-indentifier>
			// 3 bits:   <device-address set with chip pins>
			// last bit: <1..read>
			TWDR = 0xa1;
			TWCR = (1 << TWINT) | (1 << TWEN);
			while (!(TWCR & (1 << TWINT)));
			if (TW_STATUS == TW_MR_SLA_NACK) {TWCR = (1 << TWINT) | (1 << TWEN) | (1 << TWSTO); break;}
			if (TW_STATUS == TW_MR_ARB_LOST) continue;
			if (TW_STATUS != TW_MR_SLA_ACK) {TWCR = (1 << TWINT) | (1 << TWEN) | (1 << TWSTO); return 0;}

			// start read transmission
			TWCR = (1 << TWINT) | (1 << TWEN);
			while (!(TWCR & (1 << TWINT)));

			if ( (TW_STATUS == TW_MR_DATA_NACK) || (TW_STATUS == TW_MR_DATA_ACK) ) {
				*buf = TWDR;	// read is OK
				TWCR = (1 << TWINT) | (1 << TWEN) | (1 << TWSTO);	// send stop condition
				return 1;
			} else {
				TWCR = (1 << TWINT) | (1 << TWEN) | (1 << TWSTO);	// send stop condition
				return 0;
			}
			
		} while (1);
			
	} while (1);
	
}

int xeeprom_blockread(uint16_t startaddr, uint16_t noofbytes)
{
	uint8_t n, p;

	if (noofbytes == 0) return 1;
	if (noofbytes > 128) return 0;
	n = 0;
	TWBR = 3;   // see datasheet p 24.4.3 - gives 400kHz with 16Mhz clock
	TWSR = 1; 

	do {
		if (n++ >= MAX_TRIES_EEPROM) return 0;
		
		do {

			// send start cond.
			TWCR = (1 << TWINT) | (1 << TWSTA) | (1 << TWEN);
			while (!(TWCR & (1 << TWINT)));
			if (TW_STATUS == TW_MT_ARB_LOST) continue;
			if ( (TW_STATUS != TW_REP_START) && (TW_STATUS != TW_START)) { return 0;}
	
			// send 0xa0
			// 0xa0 = 1010 000 0
			// 4 bits:   <a..device-indentifier>
			// 3 bits:   <device-address set with chip pins>
			// last bit: <0..write>
			TWDR = 0xa0;
			TWCR = (1 << TWINT) | (1 << TWEN);
			while (!(TWCR & (1 << TWINT)));
			if (TW_STATUS == TW_MT_SLA_NACK) break;
			if (TW_STATUS == TW_MT_ARB_LOST) continue;
			if (TW_STATUS != TW_MT_SLA_ACK) {TWCR = (1 << TWINT) | (1 << TWEN) | (1 << TWSTO); return 0;}

			// send high 8 bits of eeaddr 
			TWDR = startaddr / 256;
			TWCR = (1 << TWINT) | (1 << TWEN);
			while (!(TWCR & (1 << TWINT)));
			if (TW_STATUS == TW_MT_DATA_NACK) {TWCR = (1 << TWINT) | (1 << TWEN) | (1 << TWSTO); break;}
			if (TW_STATUS == TW_MT_ARB_LOST) continue;
			if (TW_STATUS != TW_MT_DATA_ACK) {TWCR = (1 << TWINT) | (1 << TWEN) | (1 << TWSTO); return 0;}

			// send low 8 bits of eeaddr
			TWDR = startaddr % 256;
			TWCR = (1 << TWINT) | (1 << TWEN);
			while (!(TWCR & (1 << TWINT)));
			if (TW_STATUS == TW_MT_DATA_NACK) {TWCR = (1 << TWINT) | (1 << TWEN) | (1 << TWSTO); break;}
			if (TW_STATUS == TW_MT_ARB_LOST) continue;
			if (TW_STATUS != TW_MT_DATA_ACK) {TWCR = (1 << TWINT) | (1 << TWEN) | (1 << TWSTO); return 0;}

			// send start cond.
			TWCR = (1 << TWINT) | (1 << TWSTA) | (1 << TWEN);
			while (!(TWCR & (1 << TWINT)));
			if (TW_STATUS == TW_MT_ARB_LOST) continue;
			if ( (TW_STATUS != TW_REP_START) && (TW_STATUS != TW_START)) {return 0;}

			// send 0xa1
			// 0xa1 = 1010 000 1
			// 4 bits:   <a..device-indentifier>
			// 3 bits:   <device-address set with chip pins>
			// last bit: <1..read>
			TWDR = 0xa1;
			TWCR = (1 << TWINT) | (1 << TWEN);
			while (!(TWCR & (1 << TWINT)));
			if (TW_STATUS == TW_MR_SLA_NACK) {TWCR = (1 << TWINT) | (1 << TWEN) | (1 << TWSTO); break;}
			if (TW_STATUS == TW_MR_ARB_LOST) continue;
			if (TW_STATUS != TW_MR_SLA_ACK) {TWCR = (1 << TWINT) | (1 << TWEN) | (1 << TWSTO); return 0;}

			for (p = 0; p < noofbytes; p++)
			{
				// start read transmission
				if (p == (noofbytes - 1)) TWCR = (1 << TWINT) | (1 << TWEN);
				else TWCR = (1 << TWINT) | (1 << TWEN) | (1 << TWEA);
				while (!(TWCR & (1 << TWINT)));
				
				if ( (TW_STATUS == TW_MR_DATA_NACK) || (TW_STATUS == TW_MR_DATA_ACK) ) xeebuf[p] = TWDR;
				else {TWCR = (1 << TWINT) | (1 << TWEN) | (1 << TWSTO); return 0;}	// send stop condition
			}
			TWCR = (1 << TWINT) | (1 << TWEN) | (1 << TWSTO);	// send stop condition
			return 1;
			
		} while (1);
			
	} while (1);
	
}


int xeeprom_write_byte(uint16_t eeaddr, char buf)
{
	uint8_t n;

	n = 0;
	TWBR = 3;   // see datasheet p 24.4.3 - gives 400kHz with 16Mhz clock
	TWSR = 1; 

	do {
		if (n++ >= MAX_TRIES_EEPROM) return 0;
		
		do {

			// start cond.
			TWCR = (1 << TWINT) | (1 << TWSTA) | (1 << TWEN);
			while (!(TWCR & (1 << TWINT)));
			if (TW_STATUS == TW_MT_ARB_LOST) continue;
			if ( (TW_STATUS != TW_REP_START) && (TW_STATUS != TW_START)) {return 0;}

			// send 0xa0
			// 0xa0 = 1010 000 0
			// 4 bits:   <a..device-indentifier>
			// 3 bits:   <device-address set with chip pins>
			// last bit: <0..write>
			TWDR = 0xa0;
			TWCR = (1 << TWINT) | (1 << TWEN);
			while (!(TWCR & (1 << TWINT)));
			if (TW_STATUS == TW_MT_SLA_NACK) break;
			if (TW_STATUS == TW_MT_ARB_LOST) continue;
			if (TW_STATUS != TW_MT_SLA_ACK) {TWCR = (1 << TWINT) | (1 << TWEN) | (1 << TWSTO); return 0;}

			// send high 8 bits of eeaddr 
			TWDR = eeaddr / 256;
			TWCR = (1 << TWINT) | (1 << TWEN);
			while (!(TWCR & (1 << TWINT)));
			if (TW_STATUS == TW_MT_DATA_NACK) {TWCR = (1 << TWINT) | (1 << TWEN) | (1 << TWSTO); break;}
			if (TW_STATUS == TW_MT_ARB_LOST) continue;
			if (TW_STATUS != TW_MT_DATA_ACK) {TWCR = (1 << TWINT) | (1 << TWEN) | (1 << TWSTO); return 0;}

			// send low 8 bits of eeaddr 
			TWDR = eeaddr % 256;
			TWCR = (1 << TWINT) | (1 << TWEN);
			while (!(TWCR & (1 << TWINT)));
			if (TW_STATUS == TW_MT_DATA_NACK) {TWCR = (1 << TWINT) | (1 << TWEN) | (1 << TWSTO); break;}
			if (TW_STATUS == TW_MT_ARB_LOST) continue;
			if (TW_STATUS != TW_MT_DATA_ACK) {TWCR = (1 << TWINT) | (1 << TWEN) | (1 << TWSTO); return 0;}

			// put byte into data register and start transmission
			TWDR = buf;
			TWCR = (1 << TWINT) | (1 << TWEN);
			while (!(TWCR & (1 << TWINT)));
			if (TW_STATUS != TW_MT_DATA_ACK) {TWCR = (1 << TWINT) | (1 << TWEN) | (1 << TWSTO); return 0;}

			// send stop condition
			TWCR = (1 << TWINT) | (1 << TWEN) | (1 << TWSTO);
			return 1;

		} while (1);

	} while (1);

}


int xeeprom_blockwrite(uint16_t startaddr, uint16_t noofbytes)
{
	uint16_t addr;
	uint8_t bytesleft, bufindex;
	uint8_t n, p, count;
	
	if (noofbytes == 0) return 1;
	if (noofbytes > 128) return 0;
	addr = startaddr;
	bytesleft = noofbytes;
	bufindex = 0;
	n = 0;
	TWBR = 3;   // see datasheet p 24.4.3 - gives 400kHz with 16Mhz clock
	TWSR = 1; 

	do {
		if (n++ >= MAX_TRIES_EEPROM) return 0;
		
		do {
			// start cond.
			TWCR = (1 << TWINT) | (1 << TWSTA) | (1 << TWEN);
			while (!(TWCR & (1 << TWINT)));
			if (TW_STATUS == TW_MT_ARB_LOST) continue;
			if ( (TW_STATUS != TW_REP_START) && (TW_STATUS != TW_START)) {return 0;}

			// send 0xa0
			// 0xa0 = 1010 000 0
			// 4 bits:   <a..device-indentifier>
			// 3 bits:   <device-address set with chip pins>
			// last bit: <0..write>
			TWDR = 0xa0;
			TWCR = (1 << TWINT) | (1 << TWEN);
			while (!(TWCR & (1 << TWINT)));
			if (TW_STATUS == TW_MT_SLA_NACK) break;
			if (TW_STATUS == TW_MT_ARB_LOST) continue;
			if (TW_STATUS != TW_MT_SLA_ACK) {TWCR = (1 << TWINT) | (1 << TWEN) | (1 << TWSTO); return 0;}

			// send high 8 bits of eeaddr 
			TWDR = addr / 256;
			TWCR = (1 << TWINT) | (1 << TWEN);
			while (!(TWCR & (1 << TWINT)));
			if (TW_STATUS == TW_MT_DATA_NACK) {TWCR = (1 << TWINT) | (1 << TWEN) | (1 << TWSTO); break;}
			if (TW_STATUS == TW_MT_ARB_LOST) continue;
			if (TW_STATUS != TW_MT_DATA_ACK) {TWCR = (1 << TWINT) | (1 << TWEN) | (1 << TWSTO); return 0;}

			// send low 8 bits of eeaddr 
			TWDR = addr % 256;
			TWCR = (1 << TWINT) | (1 << TWEN);
			while (!(TWCR & (1 << TWINT)));
			if (TW_STATUS == TW_MT_DATA_NACK) {TWCR = (1 << TWINT) | (1 << TWEN) | (1 << TWSTO); break;}
			if (TW_STATUS == TW_MT_ARB_LOST) continue;
			if (TW_STATUS != TW_MT_DATA_ACK) {TWCR = (1 << TWINT) | (1 << TWEN) | (1 << TWSTO); return 0;}

			if (bytesleft > 64) count = 64; else count = bytesleft;
			for (p = 0; p < count; p++) {
				// put byte into data register and start transmission
				TWDR = xeebuf[bufindex + p];
				TWCR = (1 << TWINT) | (1 << TWEN);
				while (!(TWCR & (1 << TWINT)));
				if (TW_STATUS != TW_MT_DATA_ACK) {TWCR = (1 << TWINT) | (1 << TWEN) | (1 << TWSTO); return 0;}
			}
			// send stop condition
			TWCR = (1 << TWINT) | (1 << TWEN) | (1 << TWSTO);
			addr += count;
			bufindex += count;
			bytesleft -= count;
			n = 0;
		} while (bytesleft > 0);

	} while (bytesleft > 0);

	return 1;
}


int etdReadByte(char * pch)
{
	uint8_t n = 0;

	TWBR = 18;   // see datasheet p 24.4.3 - gives 100kHz with 16Mhz clock
	TWSR = 1;

	do {
		if (n++ > MAX_TRIES_TFT) return 0;
		
		do {

			// send start cond.
			TWCR = (1 << TWINT) | (1 << TWSTA) | (1 << TWEN);
			while (!(TWCR & (1 << TWINT)));
			if (TW_STATUS == TW_MT_ARB_LOST) continue;
			if ( (TW_STATUS != TW_REP_START) && (TW_STATUS != TW_START)) {return 0;}
			// send 0xDF
			// 0xdf = 1101 111 1
			// 4 bits:   Base address $D0 (from BA0=1  BA1=1  BA2=1)
			// 3 bits:   Slave address 110 (from SA0=0  SA1=1  SA2=1)
			// last bit: <1..read>
			TWDR = (TFTA_I2C_ADDRESS << 1) + 1;
			TWCR = (1 << TWINT) | (1 << TWEN);
			while (!(TWCR & (1 << TWINT)));
			if (TW_STATUS == TW_MR_SLA_NACK) break;
			if (TW_STATUS == TW_MR_ARB_LOST) continue;
			if (TW_STATUS != TW_MR_SLA_ACK) {TWCR = (1 << TWINT) | (1 << TWEN) | (1 << TWSTO); return 0;}

			//6us delay
			__asm__ volatile (
				"ldi r24,32 \r\n"
				"1: dec r24 \r\n"
				"brne 1b \r\n"
				: /* no outputs */
				: /* no inputs */
				: "r24"
			);
			// start read transmission
			TWCR = (1 << TWINT) | (1 << TWEN);
			while (!(TWCR & (1 << TWINT)));
			if ( (TW_STATUS == TW_MR_DATA_NACK) || (TW_STATUS == TW_MR_DATA_ACK) ) {
				*pch = TWDR;			// read is OK
				TWCR = (1 << TWINT) | (1 << TWEN) | (1 << TWSTO);	// send stop condition
				return 1;
			} else {
				TWCR = (1 << TWINT) | (1 << TWEN) | (1 << TWSTO);	// send stop condition
				return 0;
			}
			
		} while (1);
			
	} while (1);
	
}


int etdBlockRead(uint16_t noofbytes)
{
	uint8_t n = 0;
	uint8_t i;

	if (noofbytes == 0) return 1;
	if (noofbytes > 128) return 0;
	TWBR = 18;   // see datasheet p 24.4.3 - gives 100kHz with 16Mhz clock
	TWSR = 1;

	do {
		if (n++ > MAX_TRIES_TFT) return 0;
		
		do {
			// send start cond.
			TWCR = (1 << TWINT) | (1 << TWSTA) | (1 << TWEN);
			while (!(TWCR & (1 << TWINT)));
			if (TW_STATUS == TW_MT_ARB_LOST) continue;
			if ( (TW_STATUS != TW_REP_START) && (TW_STATUS != TW_START)) { return 0;}
			// send 0xDF
			// 0xdf = 1101 111 1
			// 4 bits:   Base address $D0 (from BA0=1  BA1=1  BA2=1)
			// 3 bits:   Slave address 111 (from SA0=1  SA1=1  SA2=1)
			// last bit: <1..read>
			TWDR = (TFTA_I2C_ADDRESS << 1) + 1;
			TWCR = (1 << TWINT) | (1 << TWEN);
			while (!(TWCR & (1 << TWINT)));
			if (TW_STATUS == TW_MR_SLA_NACK) break;
			if (TW_STATUS == TW_MR_ARB_LOST) continue;
			if (TW_STATUS != TW_MR_SLA_ACK) {TWCR = (1 << TWINT) | (1 << TWEN) | (1 << TWSTO); return 0;}

			for (i = 0; i < noofbytes; i++)
			{
				// 6us delay
				__asm__ volatile (
					"ldi r24,32 \r\n"
					"1: dec r24 \r\n"
					"brne 1b \r\n"
					: /* no outputs */
					: /* no inputs */
					: "r24"
				);
				// start read transmission
				if (i == (noofbytes - 1)) TWCR = (1 << TWINT) | (1 << TWEN);
				else TWCR = (1 << TWINT) | (1 << TWEN) | (1 << TWEA);
				while (!(TWCR & (1 << TWINT)));
				if ( (TW_STATUS == TW_MR_DATA_NACK) || (TW_STATUS == TW_MR_DATA_ACK) ) I2Cbuf[i] = TWDR;
				else {TWCR = (1 << TWINT) | (1 << TWEN) | (1 << TWSTO); return 0;}	// send stop condition
			}
			TWCR = (1 << TWINT) | (1 << TWEN) | (1 << TWSTO);	// send stop condition
			return 1;
			
		} while (1);
			
	} while (1);
	
}


int etdBlockWrite(uint16_t noofbytes)
{
	uint8_t n = 0;
	uint8_t p;
	
	if (noofbytes == 0) return 1;
	if (noofbytes > 255) return 0;
	TWBR = 18;   // see datasheet p 24.4.3 - gives 100kHz with 16Mhz clock
	TWSR = 1;

	do {
		if (n++ > MAX_TRIES_TFT) return 0;
		
		do {
			// start cond.
			TWCR = (1 << TWINT) | (1 << TWSTA) | (1 << TWEN);
			while (!(TWCR & (1 << TWINT)));
			if (TW_STATUS == TW_MT_ARB_LOST) continue;
			if ( (TW_STATUS != TW_REP_START) && (TW_STATUS != TW_START)) {return 0;}

			// send 0xDE
			// 0xde = 1101 111 0   
			// 4 bits:   Base address $D0 (from BA0=1  BA1=1  BA2=1)
			// 3 bits:   Slave address 111 (from SA0=1  SA1=1  SA2=1)
			// last bit: <0..write>

			TWDR = (TFTA_I2C_ADDRESS <<1) + 0;
			TWCR = (1 << TWINT) | (1 << TWEN);
			while (!(TWCR & (1 << TWINT)));
			if (TW_STATUS == TW_MT_SLA_NACK) break;
			if (TW_STATUS == TW_MT_ARB_LOST) continue;
			if (TW_STATUS != TW_MT_SLA_ACK) {TWCR = (1 << TWINT) | (1 << TWEN) | (1 << TWSTO); return 0;}

			for (p = 0; p < noofbytes; p++)
			{
				// put byte into data register and start transmission
				TWDR = I2Cbuf[p];
				TWCR = (1 << TWINT) | (1 << TWEN);
				while (!(TWCR & (1 << TWINT)));
				if (TW_STATUS != TW_MT_DATA_ACK) {TWCR = (1 << TWINT) | (1 << TWEN) | (1 << TWSTO); return 0;}
			}
			// send stop condition
			TWCR = (1 << TWINT) | (1 << TWEN) | (1 << TWSTO);
			return 1;

		} while (1);
			
	} while (1);
	
}


int icReadByte(char * pch, uint8_t IC2Address)
{
	uint8_t n = 0;

	TWBR = 3;   // see datasheet p 24.4.3 - gives 400kHz with 16Mhz clock
	TWSR = 1; 

	do {
		if (n++ > MAX_TRIES_I2C) return 0;
		
		do {

			// send start cond.
			TWCR = (1 << TWINT) | (1 << TWSTA) | (1 << TWEN);
			while (!(TWCR & (1 << TWINT)));
			if (TW_STATUS == TW_MT_ARB_LOST) continue;
			if ( (TW_STATUS != TW_REP_START) && (TW_STATUS != TW_START)) {return 0;}
			// send control
			// 4 bits:   Base address
			// 3 bits:   Slave address
			// last bit: 1 = read
			TWDR = IC2Address | 0x01; // bit 0 = 1 for read

			TWCR = (1 << TWINT) | (1 << TWEN);
			while (!(TWCR & (1 << TWINT)));
			if (TW_STATUS == TW_MR_SLA_NACK) break;
			if (TW_STATUS == TW_MR_ARB_LOST) continue;
			if (TW_STATUS != TW_MR_SLA_ACK) {TWCR = (1 << TWINT) | (1 << TWEN) | (1 << TWSTO); return 0;}
			// start read transmission
			TWCR = (1 << TWINT) | (1 << TWEN);
			while (!(TWCR & (1 << TWINT)));
			if ( (TW_STATUS == TW_MR_DATA_NACK) || (TW_STATUS == TW_MR_DATA_ACK) ) {
				*pch = TWDR;			// read is OK
				TWCR = (1 << TWINT) | (1 << TWEN) | (1 << TWSTO);	// send stop condition
				return 1;
			} else {
				TWCR = (1 << TWINT) | (1 << TWEN) | (1 << TWSTO);	// send stop condition
				return 0;
			}
			
		} while (1);
			
	} while (1);
	
}


int icBlockRead(uint16_t noofbytes, uint8_t IC2Address)
{
	uint8_t n = 0;
	uint8_t i;

	if (noofbytes == 0) return 1;
	if (noofbytes > 128) return 0;
	TWBR = 3;   // see datasheet p 24.4.3 - gives 400kHz with 16Mhz clock
	TWSR = 1; 

	do {
		if (n++ > MAX_TRIES_I2C) return 0;
		
		do {
			// send start cond.
			TWCR = (1 << TWINT) | (1 << TWSTA) | (1 << TWEN);
			while (!(TWCR & (1 << TWINT)));
			if (TW_STATUS == TW_MT_ARB_LOST) continue;
			if ( (TW_STATUS != TW_REP_START) && (TW_STATUS != TW_START)) { return 0;}
			// send control
			// 4 bits:   Base address
			// 3 bits:   Slave address
			// last bit: 1 = read
			TWDR = IC2Address | 0x01; // bit 0 = 1 for read
			TWCR = (1 << TWINT) | (1 << TWEN);
			while (!(TWCR & (1 << TWINT)));
			if (TW_STATUS == TW_MR_SLA_NACK) break;
			if (TW_STATUS == TW_MR_ARB_LOST) continue;
			if (TW_STATUS != TW_MR_SLA_ACK) {TWCR = (1 << TWINT) | (1 << TWEN) | (1 << TWSTO); return 0;}

			for (i = 0; i < noofbytes; i++)
			{
				// start read transmission
				if (i == (noofbytes - 1)) TWCR = (1 << TWINT) | (1 << TWEN);
				else TWCR = (1 << TWINT) | (1 << TWEN) | (1 << TWEA);
				while (!(TWCR & (1 << TWINT)));
				if ( (TW_STATUS == TW_MR_DATA_NACK) || (TW_STATUS == TW_MR_DATA_ACK) ) I2Cbuf[i] = TWDR;
				else {TWCR = (1 << TWINT) | (1 << TWEN) | (1 << TWSTO); return 0;}	// send stop condition
			}
			TWCR = (1 << TWINT) | (1 << TWEN) | (1 << TWSTO);	// send stop condition
			return 1;
			
		} while (1);
			
	} while (1);
	
}


int icBlockWrite(uint16_t noofbytes, uint8_t IC2Address)
{
	uint8_t n = 0;
	uint8_t p;
	
	if (noofbytes == 0) return 1;
	if (noofbytes > 255) return 0;
	TWBR = 3;   // see datasheet p 24.4.3 - gives 400kHz with 16Mhz clock
	TWSR = 1; 

	do {
		if (n++ > MAX_TRIES_I2C) return 0;
		
		do {
			// start cond.
			TWCR = (1 << TWINT) | (1 << TWSTA) | (1 << TWEN);
			while (!(TWCR & (1 << TWINT)));
			if (TW_STATUS == TW_MT_ARB_LOST) continue;
			if ( (TW_STATUS != TW_REP_START) && (TW_STATUS != TW_START)) {return 0;}
			// send control
			// 4 bits:   Base address
			// 3 bits:   Slave address
			// last bit: 0 = write
			TWDR = IC2Address; // bit 0 = 0 for write
			TWCR = (1 << TWINT) | (1 << TWEN);
			while (!(TWCR & (1 << TWINT)));
			if (TW_STATUS == TW_MT_SLA_NACK) break;
			if (TW_STATUS == TW_MT_ARB_LOST) continue;
			if (TW_STATUS != TW_MT_SLA_ACK) {TWCR = (1 << TWINT) | (1 << TWEN) | (1 << TWSTO); return 0;}

			for (p = 0; p < noofbytes; p++)
			{
				// put byte into data register and start transmission
				TWDR = I2Cbuf[p];
				TWCR = (1 << TWINT) | (1 << TWEN);
				while (!(TWCR & (1 << TWINT)));
				if (TW_STATUS != TW_MT_DATA_ACK) {TWCR = (1 << TWINT) | (1 << TWEN) | (1 << TWSTO); return 0;}
			}
			// send stop condition
			TWCR = (1 << TWINT) | (1 << TWEN) | (1 << TWSTO);
			return 1;

		} while (1);
			
	} while (1);
	
}



////////////////////// RTC TWI routines //////////////////

// Plese note that the ISL1220 I2C address is fixed at 0xDE
// ie the chip has no address select bits


int rtcReadByte(uint8_t addr, uint8_t *buf)
{
	uint8_t n;
	n = 0;
	TWBR = 3;   // see datasheet p 25.5.1 - gives 38kHz with 1.8432Mhz clock
	TWSR = 1; 
	do {
		if (n++ >= MAX_TRIES_RTC) return 0;
		
		do {
			// send start cond.
			TWCR = (1 << TWINT) | (1 << TWSTA) | (1 << TWEN);
			while (!(TWCR & (1 << TWINT)));
			if (TW_STATUS == TW_MT_ARB_LOST) continue;
			if ( (TW_STATUS != TW_REP_START) && (TW_STATUS != TW_START)) { return 0;}


			// send I2C address
			// 7 bits:   device-identifier
			// last bit: 0 = write
			TWDR = (RTC_I2C_ADDRESS << 1) + 0;
			TWCR = (1 << TWINT) | (1 << TWEN);
			while (!(TWCR & (1 << TWINT)));
			if (TW_STATUS == TW_MT_SLA_NACK) break;
			if (TW_STATUS == TW_MT_ARB_LOST) continue;
			if (TW_STATUS != TW_MT_SLA_ACK) {TWCR = (1 << TWINT) | (1 << TWEN) | (1 << TWSTO); return 0;}

			// send addr 
			TWDR = addr;
			TWCR = (1 << TWINT) | (1 << TWEN);
			while (!(TWCR & (1 << TWINT)));
			if (TW_STATUS == TW_MT_DATA_NACK) {TWCR = (1 << TWINT) | (1 << TWEN) | (1 << TWSTO); break;}
			if (TW_STATUS == TW_MT_ARB_LOST) continue;
			if (TW_STATUS != TW_MT_DATA_ACK) {TWCR = (1 << TWINT) | (1 << TWEN) | (1 << TWSTO); return 0;}

			// send start cond.
			TWCR = (1 << TWINT) | (1 << TWSTA) | (1 << TWEN);
			while (!(TWCR & (1 << TWINT)));
			if (TW_STATUS == TW_MT_ARB_LOST) continue;
			if ( (TW_STATUS != TW_REP_START) && (TW_STATUS != TW_START)) {return 0;}

			// send I2C address
			// 7 bits:   device-identifier
			// last bit: 1 = read
			TWDR = (RTC_I2C_ADDRESS << 1) + 1;
			TWCR = (1 << TWINT) | (1 << TWEN);
			while (!(TWCR & (1 << TWINT)));
			if (TW_STATUS == TW_MR_SLA_NACK) {TWCR = (1 << TWINT) | (1 << TWEN) | (1 << TWSTO); break;}
			if (TW_STATUS == TW_MR_ARB_LOST) continue;
			if (TW_STATUS != TW_MR_SLA_ACK) {TWCR = (1 << TWINT) | (1 << TWEN) | (1 << TWSTO); return 0;}

			// start read transmission
			TWCR = (1 << TWINT) | (1 << TWEN);
			while (!(TWCR & (1 << TWINT)));

			if ( (TW_STATUS == TW_MR_DATA_NACK) || (TW_STATUS == TW_MR_DATA_ACK) ) {
				*buf = TWDR;	// read is OK
				TWCR = (1 << TWINT) | (1 << TWEN) | (1 << TWSTO);	// send stop condition
				return 1;
			} else {
				TWCR = (1 << TWINT) | (1 << TWEN) | (1 << TWSTO);	// send stop condition
				return 0;
			}
			
		} while (1);
			
	} while (1);
	
}
/*
int rtcBlockRead(uint8_t startaddr, uint8_t noofbytes)
{
	uint8_t n, p;
  
	if (noofbytes > 7) {return 0;} // only one secion at a time
	if (noofbytes == 0) {return 1;}
	n = 0;
	TWBR = 3;   // see datasheet p 24.4.3 - gives 400kHz with 16Mhz clock
	TWSR = 1; 

	do {
		if (n++ >= MAX_TRIES_RTC) return 0;
		
		do {

			// send start cond.
			TWCR = (1 << TWINT) | (1 << TWSTA) | (1 << TWEN);
			while (!(TWCR & (1 << TWINT)));
			if (TW_STATUS == TW_MT_ARB_LOST) continue;
			if ( (TW_STATUS != TW_REP_START) && (TW_STATUS != TW_START)) { return 0;}
	
			// send I2C address
			// 7 bits:   device-identifier
			// last bit: 0 = write
			TWDR = (RTC_I2C_ADDRESS << 1) + 0;

			TWCR = (1 << TWINT) | (1 << TWEN);
			while (!(TWCR & (1 << TWINT)));
			if (TW_STATUS == TW_MT_SLA_NACK) break;
			if (TW_STATUS == TW_MT_ARB_LOST) continue;
			if (TW_STATUS != TW_MT_SLA_ACK) {TWCR = (1 << TWINT) | (1 << TWEN) | (1 << TWSTO); return 0;}
 
			// send startaddress
			TWDR = startaddr;
			TWCR = (1 << TWINT) | (1 << TWEN);
			while (!(TWCR & (1 << TWINT)));
			if (TW_STATUS == TW_MT_DATA_NACK) {TWCR = (1 << TWINT) | (1 << TWEN) | (1 << TWSTO); break;}
			if (TW_STATUS == TW_MT_ARB_LOST) continue;
			if (TW_STATUS != TW_MT_DATA_ACK) {TWCR = (1 << TWINT) | (1 << TWEN) | (1 << TWSTO); return 0;}

			if (TW_STATUS != TW_MT_DATA_ACK) {TWCR = (1 << TWINT) | (1 << TWEN) | (1 << TWSTO); return 0;}

			// send start cond.
			TWCR = (1 << TWINT) | (1 << TWSTA) | (1 << TWEN);
			while (!(TWCR & (1 << TWINT)));
			if (TW_STATUS == TW_MT_ARB_LOST) continue;
			if ( (TW_STATUS != TW_REP_START) && (TW_STATUS != TW_START)) {return 0;}

			// send I2C address
			// 7 bits:   device-identifier
			// last bit: 1 = read
			TWDR = (RTC_I2C_ADDRESS << 1) + 1;

			TWCR = (1 << TWINT) | (1 << TWEN);
			while (!(TWCR & (1 << TWINT)));
			if (TW_STATUS == TW_MR_SLA_NACK) {TWCR = (1 << TWINT) | (1 << TWEN) | (1 << TWSTO); break;}
			if (TW_STATUS == TW_MR_ARB_LOST) continue;
			if (TW_STATUS != TW_MR_SLA_ACK) {TWCR = (1 << TWINT) | (1 << TWEN) | (1 << TWSTO); return 0;}

			for (p = 0; p < noofbytes; p++)
			{
				// start read transmission
				if (p == (noofbytes - 1)) TWCR = (1 << TWINT) | (1 << TWEN);
				else TWCR = (1 << TWINT) | (1 << TWEN) | (1 << TWEA);
				while (!(TWCR & (1 << TWINT)));
				
				if ( (TW_STATUS == TW_MR_DATA_NACK) || (TW_STATUS == TW_MR_DATA_ACK) ) rtcbuf[p] = TWDR;
				else {TWCR = (1 << TWINT) | (1 << TWEN) | (1 << TWSTO); return 0;}	// send stop condition
			}
			TWCR = (1 << TWINT) | (1 << TWEN) | (1 << TWSTO);	// send stop condition
			return 1;
			
		} while (1);
			
	} while (1);
	
}

*/


int rtcWriteByte(uint8_t addr, uint8_t buf)
{
	uint8_t n;

	n = 0;
	TWBR = 3;   // see datasheet p 24.4.3 - gives 400kHz with 16Mhz clock
	TWSR = 1; 

	do {
		if (n++ >= MAX_TRIES_RTC) return 0;
		
		do {

			// start cond.
			TWCR = (1 << TWINT) | (1 << TWSTA) | (1 << TWEN);
			while (!(TWCR & (1 << TWINT)));
			if (TW_STATUS == TW_MT_ARB_LOST) continue;
			if ( (TW_STATUS != TW_REP_START) && (TW_STATUS != TW_START)) {return 0;}

			// send I2C address
			// 7 bits:   device-identifier
			// last bit: 0 = write
			TWDR = (RTC_I2C_ADDRESS << 1) + 0;

			TWCR = (1 << TWINT) | (1 << TWEN);
			while (!(TWCR & (1 << TWINT)));
			if (TW_STATUS == TW_MT_SLA_NACK) break;
			if (TW_STATUS == TW_MT_ARB_LOST) continue;
			if (TW_STATUS != TW_MT_SLA_ACK) {TWCR = (1 << TWINT) | (1 << TWEN) | (1 << TWSTO); return 0;}

			// send addr 
			TWDR = addr;
			TWCR = (1 << TWINT) | (1 << TWEN);
			while (!(TWCR & (1 << TWINT)));
			if (TW_STATUS == TW_MT_DATA_NACK) {TWCR = (1 << TWINT) | (1 << TWEN) | (1 << TWSTO); break;}
			if (TW_STATUS == TW_MT_ARB_LOST) continue;
			if (TW_STATUS != TW_MT_DATA_ACK) {TWCR = (1 << TWINT) | (1 << TWEN) | (1 << TWSTO); return 0;}

			// put byte into data register and start transmission
			TWDR = buf;
			TWCR = (1 << TWINT) | (1 << TWEN);
			while (!(TWCR & (1 << TWINT)));
			if (TW_STATUS != TW_MT_DATA_ACK) {TWCR = (1 << TWINT) | (1 << TWEN) | (1 << TWSTO); return 0;}

			// send stop condition
			TWCR = (1 << TWINT) | (1 << TWEN) | (1 << TWSTO);
			return 1;

		} while (1);

	} while (1);

}


int rtcBlockWrite(uint8_t startaddr, uint8_t noofbytes)
{
	uint16_t addr;
	uint8_t bytesleft, bufindex;
	uint8_t n, p, count;
	
	if (noofbytes > 6) {return 0;}
	if (noofbytes == 0) {return 0;}
	addr = startaddr;
	bytesleft = noofbytes;
	bufindex = 0;
	n = 0;
	TWBR = 3;   // see datasheet p 21.5.2 - gives 38kHz with 1.8432Mhz clock (RTC read or write take 2.6mS = ok)
	TWSR = 1; 

	do {
		if (n++ >= MAX_TRIES_RTC) return 0;
		
		do {
			// start cond.
			TWCR = (1 << TWINT) | (1 << TWSTA) | (1 << TWEN);
			while (!(TWCR & (1 << TWINT)));
			if (TW_STATUS == TW_MT_ARB_LOST) continue;
			if ( (TW_STATUS != TW_REP_START) && (TW_STATUS != TW_START)) {return 0;}

			// send I2C address
			// 7 bits:   device-identifier
			// last bit: 0 = write
			TWDR = (RTC_I2C_ADDRESS << 1) + 0;

			TWCR = (1 << TWINT) | (1 << TWEN);
			while (!(TWCR & (1 << TWINT)));
			if (TW_STATUS == TW_MT_SLA_NACK) break;
			if (TW_STATUS == TW_MT_ARB_LOST) continue;
			if (TW_STATUS != TW_MT_SLA_ACK) {TWCR = (1 << TWINT) | (1 << TWEN) | (1 << TWSTO); return 0;}

			// send address 
			TWDR = addr;
			TWCR = (1 << TWINT) | (1 << TWEN);
			while (!(TWCR & (1 << TWINT)));
			if (TW_STATUS == TW_MT_DATA_NACK) {TWCR = (1 << TWINT) | (1 << TWEN) | (1 << TWSTO); break;}
			if (TW_STATUS == TW_MT_ARB_LOST) continue;
			if (TW_STATUS != TW_MT_DATA_ACK) {TWCR = (1 << TWINT) | (1 << TWEN) | (1 << TWSTO); return 0;}



			if (bytesleft > 64) count = 64; else count = bytesleft;
			for (p = 0; p < count; p++) {
				// put byte into data register and start transmission
				TWDR = rtcbuf[bufindex + p];
				TWCR = (1 << TWINT) | (1 << TWEN);
				while (!(TWCR & (1 << TWINT)));
				if (TW_STATUS != TW_MT_DATA_ACK) {TWCR = (1 << TWINT) | (1 << TWEN) | (1 << TWSTO); return 0;}
			}
			// send stop condition
			TWCR = (1 << TWINT) | (1 << TWEN) | (1 << TWSTO);
			addr += count;
			bufindex += count;
			bytesleft -= count;
			n = 0;
		} while (bytesleft > 0);

	} while (bytesleft > 0);

	return 1;
}

