/*
 *****    homebrew-radios.net   *****

 *****     Project TrxAVR       *****


 taStarDSP.c    program file
 
Support Picastar DSP/IF unit
*/


#include "taGlobal.h"
#include <math.h>
#include "taTWI.h"
#include "taIntEeprom.h"
#include "TrxAVR.h"
#include "taStarDSP.h"
#include "taCharLCD.h"
#include "ta24LC512.h"
#include "taFT245R.h"
#include "taStarControl.h"
#include "taStarDSP.h"
#include "taProgmem.h"
#include "taDDS.h"
#include "taSWR.h"
#include "taSmeter.h"
#include "taEncoder.h"
#include "taGraphicsDriverEA320.h"
#include "taGraphicsEA320.h"
#include "taCAT.h"
#include "taZ_StarDSP.h"
#include "taZ_StarControl.h"
#include "taEATFTcommands.h"
#include "taEATFT.h"
#include "taEEPROM.h"		// for EEMEM definitions

//Rx0 baud rate
#define BAUD2 9600
#define MYUBRR_2 (Xtal/16/BAUD2-1)

#define BufTx2Size 12 
#define BufRx2Size 20

#define ParamLedPORT PORTH
#define ParamLedRed 2
#define ParamLedGreen 3

#define LayerTR 1   // all layer graphic , all text uses graphics
#define LayerRx 2
#define LayerTx 3


/*		Moved to taEEPROM.h
uint8_t EEMEM EE_CalCurve[42];
*/

static uint8_t CalCurve[42];  // resides in RAM for speed
static char BufTx2[BufTx2Size];
static char ParamValueString[10];
static uint8_t FirstTx2;
static uint8_t NextTx2;
static uint8_t SkipRx2;
static uint8_t FirstRx2;
static uint8_t NextRx2;
static double y;
static char TelemBufin[24];
static uint8_t TelemBuf[30];
static uint8_t TelemFlag;
static uint8_t DspCount;
static uint16_t Adc;
static uint8_t Hang;
static uint8_t Spike;
static uint16_t Dac;
static uint8_t	uartADCgain;
static uint8_t GN;
static uint8_t hs;
static uint8_t hff;
static int agcdb;
static uint16_t ADCdb;
static uint8_t TotalGain;
static uint8_t AD603gain;
static uint8_t micgain;
static uint8_t PrevTx;
static uint16_t Dev1;
static uint16_t Dev2;
static uint16_t Dev3;
static uint16_t Dev4;
static uint16_t Dev5;
static uint16_t Dev6;
static uint16_t dspdata;
static uint8_t	BarPixEA320[2][9];   // TR  and 1 to 8
static uint8_t	BarPixEATFT[2][9];   // TR  and 1 to 8
static uint8_t dBmax[9];
static uint8_t dBmin[9]; 
static uint16_t AD603pixEA320;
static uint16_t ADCpixEA320;
static uint16_t AGCpixEA320;
static uint16_t PrevAD603pixEA320;
static uint16_t PrevSparePosEA320;
static uint16_t AD603pixEATFT;
static uint16_t ADCpixEATFT;
static uint16_t AGCpixEATFT;
static uint16_t PrevAD603pixEATFT;
static uint16_t PrevSparePosEATFT;
static uint16_t PrevAGCpixEATFT;
static uint8_t smcount;
static uint16_t dspParamsAddr;
static uint16_t dspGreenParamsAddr;
static uint16_t DAC;
static uint16_t SLed;
static uint8_t SPK;
static uint8_t PreADCgain;
static uint8_t HangStuff;

uint8_t rx2;
uint8_t ParamVal;
uint8_t Muted;
uint8_t ParamColour;
uint8_t SmeterdB;
uint8_t DspMode;
uint8_t HaltDSPdisplay;
uint8_t ParamsNotRestoredMsg;


uint8_t dspReceiveCalCurve();
void dspLoadCalCurve();
uint8_t uaRx2BufCount();
uint8_t uaRx2Empty();
uint8_t uaRx2Full();
uint8_t uaTx2Empty();
uint8_t uaTx2Full();
void    dspResetTx2();
uint8_t dspTx2Empty();
uint8_t dspTx2Full();
uint8_t dspTx2BufCount();
void    dspIntEnableTx2(unsigned char tf);
uint8_t dspTransmitCharTx2(char pc);
uint8_t dspReadCharRx2(char* pc);
void    dspResetRx2();
uint8_t dspRx2Full();
uint8_t dspRx2BufCount();
void    dspIntEnableRx2(unsigned char tf);
void dspCopyBackupParams(); 
void dspLoadingBar(uint8_t bars);
uint16_t dspGetCodeFromUSB();
uint8_t dspGetGreenParam(uint8_t x);
void dspSendMonitorDataToPC();
uint8_t dspReceiveCode();
uint8_t dspReceiveParams();
void dspSend245DataToPC();
void dspPcSetParamToBackup();
void dspCopyParamsToBackup();
uint8_t dspSendParamsToPC();
uint8_t dspReceiveOneParam();
uint8_t dspReceiveParamColour();
void dspSendTelem();
uint8_t dspCalcDB(uint16_t dspdata);
void dspDisplayCalc();
void dspDoGraphicEA();
void dspDoBottomDisplay();
void dspInitBarsEA320();
void dspInitBarsEATFT();
void dspInitMaxMin();
void dspEA320GainBar();
void dspEA320VOGADgain();
void dspEA320eightBars();
void dspEATFTGainBar();
void dspEATFTVOGADgain();
void dspEATFTeightBars();



void dspInit() __attribute__((section(".highmem")));
void dspInit()
{
	UBRR2H = (unsigned char)(MYUBRR_2>>8);  // set baud rate
	UBRR2L = (unsigned char)MYUBRR_2;
  UCSR2C = 0x06;  //   asynchronous, no parity, 1 stop bit, 8 data bits 
	UCSR2B = 0x18;   /// set bits 3 and 4 to enable transmission and reception
  dspResetTx2();
	dspResetRx2();
  TelemFlag = 0;
	Muted = 0;
	DspCount = 0;
	/////////////////
  SWmonitor52 = 2;
  dspSetParam(52,2);
	/////////////////
//  mode = LSB;    //  no this is done in scStarInit
	dspLoadCalCurve();
}


void dspSetSSBfilterCentre(uint16_t widthparam) __attribute__((section(".highmem")));
void dspSetSSBfilterCentre(uint16_t widthparam)
{
	uint8_t v, c;

	v = dspGetParamValue(widthparam);
	c = (((uint16_t)v * 3) / 4) + 45;
	xeWriteByte(dspParamsAddr + 65, c);
}


uint8_t dspReceiveCalCurve() __attribute__((section(".highmem")));
uint8_t dspReceiveCalCurve()
{
	if(usbReceiveBlock(CalCurve,42,500)==1)
	{
		ie_eeprom_write_block(CalCurve, ee.EE_CalCurve, 42);	
	} 
	return 1;
}


void dspLoadCalCurve() __attribute__((section(".highmem")));
void dspLoadCalCurve()
{
  ie_eeprom_read_block(CalCurve, ee.EE_CalCurve, 42);
}


void dspLoad() __attribute__((section(".highmem")));
void dspLoad()
{
	_delay_ms(500);
	dspLoadCode();
	_delay_ms(1000);  
	ParamsNotRestoredMsg = 0;
	if (DDSparams.dpParamResetMode == 0) {
		dspRestoreAllBackupParams();
	} else {
		if ((zc == 1) || (zs == 1)) {
			strcpy(StrA,PMS(s_Params_not_restored));  // removed IJS 13.5.2010 
    		zdspMsg(StrA, 1);                         // message now in message window
		} else ParamsNotRestoredMsg = 1;  
	}
}



void dspSetParameterColour() __attribute__((section(".highmem")));
void dspSetParameterColour()
{
  dspSetLed(ParamColour);
  dspLoadParams();
}


void dspRestoreDefaultParams() __attribute__((section(".highmem")));
void dspRestoreDefaultParams()
{
  dspCopyBackupParams();
	dspLoadParams();
}  



void dspSetLed(uint8_t colour) __attribute__((section(".highmem")));
void dspSetLed(uint8_t colour)
{
	zscDisplayParamColour(colour);
	switch(colour)
	{
	  case(pgreen):
	  {
		  ParamLedPORT |= (1 << ParamLedGreen);
			ParamLedPORT &= ~(1 << ParamLedRed);
      break;
    }
	  case(pred):
	  {
		  ParamLedPORT |= (1 << ParamLedRed);
			ParamLedPORT &= ~(1 << ParamLedGreen);
      break;
    }
	  case(pyellow):
	  {
		  ParamLedPORT |= (1 << ParamLedGreen);
			ParamLedPORT |= (1 << ParamLedRed);
      break;
    }
  }
}




// PC request DSP sub decode   eg: $$$DSP_A   $$$DSP_C   etc
void dspUsbRequest(char cmd) __attribute__((section(".highmem")));
void dspUsbRequest(char cmd)
{
	switch (cmd) {
		case 'A': dspSendMonitorDataToPC(); break;    // send current set,switches etc to PC monitoring
		case 'B': dspPcSetParamToBackup(); break;
		case 'C': dspReceiveCode(); break;
		case 'G': dspSend245DataToPC(); break;
		case 'P': dspReceiveParams();xeSetTxDriveToDefault(); break; 
		case 'Q': dspReceiveOneParam(); break;
		case 'L': dspReceiveParamColour(); break;
		case 'T': dspSendTelem(); break;
		case 'S': dspSendParamsToPC();   // send SSB & CW  / G Y R to PC
		case 'V': dspReceiveCalCurve();
		case 'W': dspCopyParamsToBackup();
	}
}


///////////////////   TX2 - transmit to DSP  //////////////////


ISR(USART2_TX_vect)   // interrupt on character trasmitted on USART2 
{
	if (dspTx2Empty() == 1) return;
	char c = BufTx2[FirstTx2++];
	if (FirstTx2 >= BufTx2Size) FirstTx2 = 0;
	dspTransmitCharTx2(c);
	return;
}


void dspResetTx2() __attribute__((section(".highmem")));
void dspResetTx2()
{
	FirstTx2 = 0;
	NextTx2 = 0;
	dspIntEnableTx2(1); // enable Tx interrupt INT2
	return;
}


uint8_t dspTx2Empty() __attribute__((section(".highmem")));
uint8_t dspTx2Empty()
{
  if(FirstTx2 == NextTx2){return 1;};
	return 0;
}



uint8_t dspTx2Full() __attribute__((section(".highmem")));
uint8_t dspTx2Full()
{
  uint8_t n = NextTx2;
	if(n < FirstTx2 ){n+=BufTx2Size;};
  if((n - FirstTx2) >= (BufTx2Size-1)) return 1; // buffer full
	return 0;   // buffer not full
}


uint8_t dspTx2BufCount() __attribute__((section(".highmem")));
uint8_t dspTx2BufCount()
{
  if(NextTx2>=FirstTx2)
	{return (NextTx2-FirstTx2);}
	else
	{return (NextTx2 + BufTx2Size - FirstTx2);};  
}



void dspIntEnableTx2(unsigned char tf) __attribute__((section(".highmem")));
void dspIntEnableTx2(unsigned char tf)
{
  if (tf == 0)
	UCSR2B &= ~(1 << 6);  // disable TXCIE2 (bit 6)
	else
	UCSR2B |= (1 << 6);   // enable TXCIE2 (bit 6)
  return;
}


uint8_t dspTransmitCharTx2(char pc) __attribute__((section(".highmem")));
uint8_t dspTransmitCharTx2(char pc)
{
  int timeout = 0;
	while((UCSR2A & (1<<UDRE2))==0)   // check for ready for transmit  (UDRE2 - bit 5)
	{													 // with be set if arrived here from interrupt
		_delay_us(10);
		if(++timeout > 1000){return 0;};
	}
  UDR2 = pc;  //write char
	return 1;
}



uint8_t dspSendCharTx2(char pc) __attribute__((section(".highmem")));
uint8_t dspSendCharTx2(char pc)
{
  if(dspTx2Empty()==1) // && (UCSR0A & (1<<UDRE2)))   // bit 5 = UDRE2
	{
	  dspTransmitCharTx2(pc);
		return 1;
  }
	int timeout = 0;
	while(dspTx2Full()==1)
	{
		_delay_us(10);
		if(++timeout > 1000){return 0;};
	}
  BufTx2[NextTx2++] = pc;
	if(NextTx2 >= BufTx2Size){NextTx2 = 0;}; 
  
	return 1;

}
///////////////////   RX2 - receive from  DSP //////////////////


ISR(USART2_RX_vect)   // interrupt on character receieved on USART2
{
	if (smcount >= 24) {smcount=0; return;}
	char ch = 0;
	uint8_t r = dspReadCharRx2(&ch);
	if (r == 0) return;
	switch (smcount)
	{
		case(0): {if (ch == '~') {smcount = 1;}  return;}
		case(1): {if (ch == '~') {smcount = 2;} else {smcount = 0;} return;};
	}
	// smcount must be in range 2 - 23
	TelemBufin[smcount] = ch;
	smcount += 1;
	if (smcount == 24)
	{
		uint8_t i;
		TelemBuf[0] = '~';
		TelemBuf[1] = '~';
		for (i = 2; i < 24; i++) {TelemBuf[i] = (uint8_t)TelemBufin[i];}
		smcount = 0;
		TelemFlag = 1;
		Adc = TelemBuf[3] + 256*TelemBuf[2];
		HangStuff = TelemBuf[4];
		DAC  = TelemBuf[6]  + 256*TelemBuf[5];
		Dev1 = TelemBuf[8]  + 256*TelemBuf[7];
		Dev2 = TelemBuf[10] + 256*TelemBuf[9];
		Dev3 = TelemBuf[12] + 256*TelemBuf[11];
		Dev4 = TelemBuf[14] + 256*TelemBuf[13];
		Dev5 = TelemBuf[16] + 256*TelemBuf[15];
		Dev6 = TelemBuf[18] + 256*TelemBuf[17];
		SmeterdB = TelemBuf[19];
		SPK = TelemBuf[20];
		PreADCgain = TelemBuf[21]/16;
		SLed = (TelemBuf[21] & 0x0F)*256 + TelemBuf[22];
		GN = TelemBuf[23];
	}
	return;
}					 
  

void dspSendTelem() __attribute__((section(".highmem")));
void dspSendTelem()
{
	uint16_t swrdata;

	swrdata = FwdCentiWatts / 10;
	memcpy(&TelemBuf[24], &swrdata, 2);
	swrdata = RefCentiWatts / 10;
	memcpy(&TelemBuf[26], &swrdata, 2);
	swrdata = SWRx100;
	memcpy(&TelemBuf[28], &swrdata, 2);
	usbSendBlock(TelemBuf, 30, 1000); 
}




uint8_t dspReadCharRx2(char* pc) __attribute__((section(".highmem")));
uint8_t dspReadCharRx2(char* pc)
{
  int timeout = 0;
	while( !(UCSR2A & 0x80))
	{
    timeout += 1;
		_delay_us(10);
		if(++timeout > 1000){return 0;};
	}
  *pc = UDR2;
	return 1;
}




void dspResetRx2() __attribute__((section(".highmem")));
void dspResetRx2()
{
	SkipRx2 = 0;
	FirstRx2 = 0;
	NextRx2 = 0;
	smcount = 0;
	dspIntEnableRx2(1); // enable Rx interrupt INT2
	return;
}



uint8_t dspRx2Empty() __attribute__((section(".highmem")));
uint8_t dspRx2Empty()
{
  if(FirstRx2 == NextRx2){return 1;};
	return 0;
}



uint8_t dspRx2Full() __attribute__((section(".highmem")));
uint8_t dspRx2Full()
{
  uint8_t n = NextRx2;
	if(n < FirstRx2 ){n+=BufRx2Size;};
  if((n - FirstRx2) >= (BufRx2Size-1)) return 1; // buffer full
	return 0;   // buffer not full
}


uint8_t dspRx2BufCount() __attribute__((section(".highmem")));
uint8_t dspRx2BufCount()
{
  if(NextRx2>=FirstRx2)
	{return (NextRx2-FirstRx2);}
	else
	{return (NextRx2 + BufRx2Size - FirstRx2);};  
}



void dspIntEnableRx2(unsigned char tf) __attribute__((section(".highmem")));
void dspIntEnableRx2(unsigned char tf)
{
  if (tf == 0)
	UCSR2B &= ~(1 << 7);  // disable RXCIE2 (bit 7)
	else
	UCSR2B |= (1 << 7);   // enable RXCIE2 (bit 7)
  return;
}

///////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////

void dspSend245DataToPC() __attribute__((section(".highmem")));
void dspSend245DataToPC()
{
	uint8_t i, buf[90];
	uint16_t addr;

	addr = xeGetDspParamsAddress((5 * (Mode / 2)) + ParamColour - 1);
	for (i = 0; i <= 88; i++) xeReadByte(addr+i+11, &buf[i]);
	usbSendBlock(buf, 90, 1000);
}


void dspPcSetParamToBackup() __attribute__((section(".highmem")));
void dspPcSetParamToBackup()
{
	uint8_t buf[3], paramno, colour, mode, val, gflag;
	uint16_t Addr, BackupAddr, GreenAddr;

	if (usbReceiveBlock(&buf[0], 3, 1000) == 0) return;
	paramno = buf[0];
	colour = buf[1];
	mode = buf[2];
	GreenAddr = xeGetDspParamsAddress((5 * (mode / 2)) + pgreen - 1);
	Addr = xeGetDspParamsAddress((5 * (mode / 2)) + colour - 1);
	BackupAddr = Addr - 0x80;
	xeReadByte(BackupAddr+paramno, &val);
	xeWriteByte(Addr+paramno, val);
	if (val == 255) return;
	if (val == 254) {
		xeReadByte(GreenAddr+paramno, &val); // use green param
		gflag = 1;
	} else gflag = 0;
	dspSendCommand(paramno, val);
	if ((ParamColour == colour) && (Mode == mode)) {
		if (paramno == 62) {SWfilter = 1; SSB_SWfilter = 1;}
		if (paramno == 63) {SWfilter = 0; SSB_SWfilter = 0;}
		if (paramno == 61) {SWfilter = 1; CW_SWfilter = 1;}
//	if (paramno == 64) {SWfilter = 0; CW_SWfilter = 0;}   // IJS Star6a9
		if (paramno == 64) {SWfilter = 0; CW_SWfilter = 1;}
		if ((paramno == 6) || (paramno == 62) || (paramno == 63)) dspSetSSBfilterCentre(63-SWfilter);
		if ((paramno >= 61) && (paramno <= 66)) zscShowFilter();
	}
	if (paramno == 66) {
		if (gflag) {
			xeWriteByte(Addr+42, 254);
			xeWriteByte(GreenAddr+42, val);
		} else {
			xeWriteByte(Addr+42, val);
		}
	}
}


void dspCopyParamsToBackup() __attribute__((section(".highmem")));
void dspCopyParamsToBackup()  // for current mode & colour
{
	uint16_t BackupAddr;

	BackupAddr = dspParamsAddr - 0x80;
	xeeprom_blockread(dspParamsAddr, 100);
	xeeprom_blockwrite(BackupAddr, 100);
}


void dspSendMonitorDataToPC() __attribute__((section(".highmem")));
void dspSendMonitorDataToPC()
{
	uint8_t buf[114];
	uint8_t i;

 	buf[0] = Transmit;
	buf[1] = SWdenoise;
	buf[2] = Mode;
	buf[3] = SWnoiseblank;
	buf[5] = SWmanualnotch;
	buf[6] = SWfilter;
	buf[8] = 0;
	buf[10] = ParamColour;
	if (Mode < 2) {
		buf[4] = SWautonotch;
		buf[7] = SWvox;
		buf[9] = SWspeechcomp;
	} else {
		buf[4] = SWqro;
		buf[7] = SWqsk;
		buf[9] = SWspot;
	}
	for (i = 11; i <= 99; i++) {buf[i] = dspGetParamValue(i);}
	buf[100] = 0;
	for (i = 1; i <= 12; i++) {buf[100+i] = EncodersPotsParams[i-1];}  // sent as userno
	usbSendBlock(buf, 113, 1000); 
}


uint8_t dspReceiveOneParam() __attribute__((section(".highmem")));
uint8_t dspReceiveOneParam()
{
	uint8_t paramno, value, buf[2];
	
	if (usbReceiveBlock(&buf[0], 2, 1000) == 0) return 0;
	paramno = buf[0];
	value = buf[1];
	switch (paramno) {
		case(1): SWdenoise = value; break; 
		case(3): SWnoiseblank = value; break; 
		case(4): if (Mode < 2) SWautonotch = value; else SWqro = value; break; 
		case(5): SWmanualnotch = value; break; 
		case(6): {
			SWfilter = value;
			if (Mode < 2) SSB_SWfilter = SWfilter;
			else CW_SWfilter = SWfilter;
			break; 
		}
		case(7): if (Mode < 2) SWvox = value; else SWqsk = value; break; 
		case(9): if (Mode < 2) SWspeechcomp = value; else SWspot = value; break; 
		case(2): {
			Mode = value; 
			dspLoadTxDrive();
			dspLoadParams();  
			enSetEncodersMaxMin();    
			zscShowMode();
			smStartSmeter();  // need 5.2 = 1 until smeter drive changed
			break;
		}	  
	}
	dspSetParam(paramno, value);
	zscShowSwitches();
	return 1;
}


uint8_t dspReceiveParamColour() __attribute__((section(".highmem")));
uint8_t dspReceiveParamColour()
{
	uint8_t buf[2];

	if (usbReceiveBlock(&buf[0], 2, 1000) == 0) return 0;
	ParamColour = buf[0];
	dspSetParameterColour();
	zscShowFilter();
	return 1;
}

//////////////////////////////////////////////////////////////////
/////////////////   DSP code and parameter reception from PC



uint16_t dspGetCodeFromUSB() __attribute__((section(".highmem")));
uint16_t dspGetCodeFromUSB()
{
	uint32_t addr, avrchecksum;
	uint16_t bc, p, remaining, bksz, ByteCount;
	uint8_t fails, done, i, b;

	// use eebuf
	if (usbReceiveBlock(&xeebuf[0], 2, 5000) == 0) return 0;
	ByteCount = 256*xeebuf[0] + xeebuf[1];
	xeWriteByte(0x7FE8, xeebuf[0]);  // store byte count
	xeWriteByte(0x7FE9, xeebuf[1]);
	if (usbReceiveBlock(&xeebuf[0], 14, 5000) == 0) return 0;
	for (p = 0; p < 13; p++) {if(xeebuf[p] != '#') return 0;}
	if (xeebuf[13] != 'x') return 0;
	ByteCount -= 14;    // #############x
	avrchecksum = 13*'#' + 'x';
	fails = 0;
	bc = 14;
	p = 0;
	addr = 0;   // block write must not cross 128 byte page boundaries
	done = 0;
	while (done == 0) {
		remaining = ByteCount - p;
		if (remaining <= 128) done = 1;
		if (remaining >= 128) bksz = 128; else bksz = remaining;
		for (i = 0; i < bksz; i++) {
			_delay_us(100);
			do {
				if (usbGetByte(&b) == 0) {
					fails++;
					_delay_ms(10);  //was 300  - Max incoming speed about 5 bytes/mS
				} else {
					fails = 0;
				}
				if (fails > 400) return 0;  // was 10 - this was too small for slow USB input
			} while (fails > 0);
			xeebuf[i] = b;
			avrchecksum = (avrchecksum + b) % 0x10000;
			bc++;
		}
		xeeprom_blockwrite(addr, bksz);
		addr += bksz;
		p += bksz;
	}
	avrchecksum = (avrchecksum + bc) % 0x10000;
	return avrchecksum;
}


uint8_t dspReceiveCode() __attribute__((section(".highmem")));
uint8_t dspReceiveCode()
{
	uint16_t chksum;

	chksum = dspGetCodeFromUSB();
	if (chksum == 0) return 0;
	usbPutByte(chksum / 256);
	usbPutByte(chksum % 256);
	return 1;
}


// All param storage starts at a 128 byte page boundary and so 
// writing 100 bytes never crosses the boundary

uint8_t dspReceiveParams() __attribute__((section(".highmem")));
uint8_t dspReceiveParams()
{
	uint8_t RP_rc, RP_i, RP_np;
	uint16_t RP_addr;

	RP_rc = 0;
	for (RP_np = 0; RP_np < 10; RP_np++) {
		if (usbReceiveBlock(&xeebuf[0], 14, 5000) == 0) {RP_rc = 10 + RP_np; break;}
		for (RP_i = 0; RP_i < 13; RP_i++) {if (xeebuf[RP_i] != '#') {RP_rc = 20 + RP_np; break;}}
		if (RP_rc) break;
		if (xeebuf[13] != 13) {RP_rc = 30 + RP_np; break;}
		for (RP_i = 0; RP_i < 10; RP_i++) xeebuf[RP_i] = 0;   // switches
		if (usbReceiveBlock(&xeebuf[10], 90, 5000) == 0) {RP_rc = 40 + RP_np; break;}
		RP_addr = xeGetDspParamsAddress(RP_np);
		xeeprom_blockwrite(RP_addr, 100);
		if ((RP_np <= 2) || ((RP_np >= 5) && (RP_np <= 7))) { // 0-2 and 5-7  - backup set
			RP_addr -= 0x80; 
			xeeprom_blockwrite(RP_addr, 100); // backup param data
		}
	}
	if (usbReceiveBlock(&xeebuf[0], 8, 5000) == 0) RP_rc += 50;  // final 8x'#'
	for (RP_i = 0; RP_i < 8; RP_i++) {if (xeebuf[RP_i] != '#') {RP_rc += 100; break;}}
	dspLoadParams();   // use the new params for current mode/colour
	// ddsResetParams(); // where else do we do this??
	ddsWriteOldestStickyVfoSlotNo(1);
	if (RP_rc) return 0; else return 1;
}


uint8_t dspSendParamsToPC() __attribute__((section(".highmem")));
uint8_t dspSendParamsToPC()
{
	uint8_t ps;
	uint8_t i;
	uint8_t np;
	uint16_t addr;
	
	for (np = 0; np <= 5; np++)   //   SSB green,yellow  then red
	{
		ps = np;
		if (np > 2) ps = np + 2;  //   need to send 0,1,2,5,6,7
		addr = xeGetDspParamsAddress(ps);    
		xeeprom_blockread(addr, 100);   // parameters in 10-99    0-9 unused switch bytes 
		for (i = 0; i <= 9; i++) {xeebuf[i] = '#';};   // ten # to sync in 0-9 switch bytes
		if (usbSendBlock(&xeebuf[0], 100, 5000) == 0) return 0;
	}
	return 1;
}


////////////////////////////////////////////////////////////////////////////////
////////////  Loading code and default parameters to DSP ////////////////

uint8_t dspLoadCode() __attribute__((section(".highmem")));
uint8_t dspLoadCode()
{
	uint32_t todspchecksum;
	uint16_t bc, p, ByteCount, addr, BytesPerBar, NextBarByteCount;
	uint8_t b, c, d, i, hcount, needdelays, sendhold, BarCount, hold[4];

	addr = 0;
	todspchecksum = (13 * '#') + 'x';
	bc = 14;
	dspSendCommand(111, 0);
	_delay_ms(1000);
	for (p = 1; p <= 3; p++) {if (dspSendCharTx2('$') == 0) return 0;} //hfr
	if (dspSendCharTx2(13) == 0) return 0;
	_delay_ms(500);
     ByteCount =  (xeGetDspCodeByteCount() - 105); //offset for 7 -> #
          if ( (ByteCount < 18000)) return 0;   
	BytesPerBar = ByteCount / 16;  // for 16 section progress bar
	NextBarByteCount = BytesPerBar;
	BarCount = 0;
	hcount = 0;
	for (p = 0; p < 0xffff; p++){ 
		if (p >= NextBarByteCount) {
			BarCount++;
			NextBarByteCount += BytesPerBar;
			zdspLoadingBar(BarCount); 
    	}
		xeReadByte(addr, &b);
    	addr++;
		xeReadByte((addr), &c);
		xeReadByte((addr + 2), &d);
		if((c == 0x23) && (b == 0x23) && (d == 0x23)){
	dspSendCharTx2(13); 
	dspSendCharTx2(13);	 
	_delay_ms(100);
	if (dspSendCharTx2('$') == 0) return 0;
	if (dspSendCharTx2('G') == 0) return 0;
	if (dspSendCharTx2('O') == 0) return 0;
	if (dspSendCharTx2(0) == 0) return 0;
	if (dspSendCharTx2(0) == 0) return 0;
	if (dspSendCharTx2(13) == 0) return 0;
	
	_delay_ms(100);
	todspchecksum = (todspchecksum + bc) % 0x10000;
	return 1;
		}
		 else  {
		sendhold = 0;
		needdelays = 0;
		if (hcount > 0) {  // delay needed before byte pattern "$DP"
			hcount++;      // so - collect and hold up to three bytes
			hold[hcount] = b;
			switch(hcount) {
				case 2: if (b != 'D') sendhold = 1; break;
				case 3: if (b == 'P') needdelays = 1; sendhold = 1; break;
			}
			if (sendhold == 1) {
				for (i = 1; i <= hcount; i++) {
			    	if (needdelays && (i == 1)) _delay_ms(300); // 10mS is too small for large $dp packets to load 4xx hfr
					dspSendCharTx2(hold[i]);
					todspchecksum = (todspchecksum + b) % 0x10000;
					bc++;
				}
				hcount = 0;
				
			}
		} else {
			if (b == '$') {
				hcount = 1;
				hold[1] = b;
			} else {
		    	dspSendCharTx2(b);
				todspchecksum = (todspchecksum + b) % 0x10000;
				bc++;
			}
		}
			
			 
    }//else


	}
return 0;
}


void dspTidySwitches() __attribute__((section(".highmem")));
void dspTidySwitches()
{

	uint8_t bl;

	SWdenoise = 0;
	SWnoiseblank = 0;
	SWautonotch = 0;
	SWmanualnotch = 0;
	SWfilter = 0;
	SSB_SWfilter = 0;
	CW_SWfilter = 1;
	SWvox = 0;
	SWspeechcomp = 0;
	SWqro = 0;
	SWqsk = 0;
	SWspot = 0;
	xeWriteByte(dspParamsAddr+1, 0);  // denoise off
	xeWriteByte(dspParamsAddr+3, 0);  // noise blank off
	xeWriteByte(dspParamsAddr+4, 0);  // auto notch off
	xeWriteByte(dspParamsAddr+5, 0);  // manual notch off
	xeWriteByte(dspParamsAddr+6, 0);  // wide filter
	xeWriteByte(dspParamsAddr+7, 0);  // vox off
	xeWriteByte(dspParamsAddr+9, 0);  // comp  off
	bl = dspSendBulkLoad(1);
	dspSendCommand(1, 0);	// denoise off
	dspSendCommand(3, 0);  // noise blank off
	dspSendCommand(4, 0);  // auto notch off
	dspSendCommand(5, 0);  // manual notch off
	dspSendCommand(6, 0);  // wide filter
	dspSendCommand(7, 0);  // vox off
	dspSendCommand(9, 0);  // comp off

	if (bl) dspSendBulkLoad(0);

}  


void dspLoadSwitches() __attribute__((section(".highmem")));
void dspLoadSwitches()
{
	uint8_t i, v, bl;

	bl = dspSendBulkLoad(1);
	for (i = 1; i <= 9; i++) {
		v = dspGetParamValue(i);
		dspSendCommand(i, v);
	}
	if (Mode < 2) dspSetSSBfilterCentre(63-v);
	if (bl) dspSendBulkLoad(0);
}


uint8_t dspSendBulkLoad(uint8_t request) __attribute__((section(".highmem")));
uint8_t dspSendBulkLoad(uint8_t request)
{
	static uint8_t BulkLoad;

	if (request) { // bulk load start request
		if (BulkLoad) {
			return 0;
		} else {
			dspSendCommand(108, 1); //    mark as bulk loading started
			BulkLoad = 1;
			return 1;
		}
	} else { // bulk load finish request
		if (BulkLoad) {
			dspSendCommand(108, 0); //    mark as bulk loading finished
			BulkLoad = 0;
			return 1;
		} else {
			return 0;
		}
	}
}


void dspSetParamAddress() __attribute__((section(".highmem")));
void dspSetParamAddress()
{
  // eg: green = 1, cwlsb = 2  ->  list index = 5
	dspParamsAddr = xeGetDspParamsAddress((5 * (Mode / 2)) + ParamColour - 1);
	dspGreenParamsAddr = xeGetDspParamsAddress((5 * (Mode / 2)) + pgreen - 1);


}


void dspLoadParams() __attribute__((section(".highmem")));
void dspLoadParams()
{
	uint8_t val, starno, bl;
    
	dspSetParamAddress();
	bl = dspSendBulkLoad(1);
	for (starno = 10; starno <= 99; starno++) {
		val = dspGetParamValue(starno);
		if (val < 255) {
			dspSendCommand(starno, val);
		}
	}
	dspSendCommand(6, SWfilter);
	if (Mode < 2) dspSetSSBfilterCentre(63-SWfilter);
	if (bl) dspSendBulkLoad(0);
}


void dspRestoreAllBackupParams() __attribute__((section(".highmem")));
void dspRestoreAllBackupParams()  // for both CW and SSB for all colours
{
	uint8_t ParamColourSave, ModeSave;

	ParamColourSave = ParamColour;
	ModeSave = Mode;
	for (ParamColour = pgreen; ParamColour <= pred; ParamColour++) {
		Mode = 0;
		dspCopyBackupParams();
		Mode = 2;
		dspCopyBackupParams();
	}
	ParamColour = ParamColourSave;
	Mode = ModeSave;
}


void dspCopyBackupParams() __attribute__((section(".highmem")));
void dspCopyBackupParams()  // for current mode & colour
{
	uint16_t Addr, BackupAddr;

	Addr = xeGetDspParamsAddress((5*(Mode/2)) + ParamColour - 1);
	BackupAddr = Addr - 0x80;
	xeeprom_blockread(BackupAddr, 100);
	xeeprom_blockwrite(Addr, 100);
}


void dspRestoreParamToBackup(uint8_t starno) __attribute__((section(".highmem")));
void dspRestoreParamToBackup(uint8_t starno)  // for current mode & colour
{
	uint8_t val;
	uint16_t BackupAddr;

	BackupAddr = dspParamsAddr - 0x80; 
	xeReadByte(BackupAddr+starno, &val);
	dspSetParam(starno, val);
}


void dspRestoreParamToBackupDontSend(uint8_t starno) __attribute__((section(".highmem")));
void dspRestoreParamToBackupDontSend(uint8_t starno)  // for current mode & colour
{
	uint8_t val;
	uint16_t Addr, BackupAddr;

	Addr = xeGetDspParamsAddress((5*(Mode/2)) + ParamColour - 1);
	BackupAddr = Addr - 0x80; 
	xeReadByte(BackupAddr+starno, &val);
	xeWriteByte(Addr+starno, val);
}


void dspLoadTxDrive() __attribute__((section(".highmem")));
void dspLoadTxDrive()
{
	uint8_t drv;
	uint16_t EEaddr;

// IJSmod 25082012  remove following line
//	if ((Mode & 0x2) == 0) EEaddr = xeGetTxDriveSSBaddr(); else EEaddr = xeGetTxDriveCWaddr();

// add following line
	EEaddr = dspParamsAddr + 0x70;      // IJSmod 25082012

	xeReadByte(EEaddr + Band, &drv);
	dspSetParam(81, drv);
}


void dspStoreTxDrive(uint8_t drv) __attribute__((section(".highmem")));
void dspStoreTxDrive(uint8_t drv)
{
 	uint16_t EEaddr;

// IJSmod 25082012  remove following line
//	if ((Mode & 0x2) == 0) EEaddr = xeGetTxDriveSSBaddr(); else EEaddr = xeGetTxDriveCWaddr();

// add following line
	EEaddr = dspParamsAddr + 0x70;    // IJSmod 25082012

	xeWriteByte(EEaddr + Band, drv);
	dspSetParam(81, drv);
}


//////////////////////////////////////////////////////////////////////////////
////////// Parameter controls /////////////////////////////////////

void dspSendCommand(uint8_t param, uint8_t data ) __attribute__((section(".highmem")));
void dspSendCommand(uint8_t param, uint8_t data )
{
	if(dspSendCharTx2('~')==0){return;};
	if(dspSendCharTx2(param)==0){return;};
	if(dspSendCharTx2(data)==0){return;};
}


void dspSetParam(uint8_t p, uint8_t v) __attribute__((section(".highmem")));
void dspSetParam(uint8_t p, uint8_t v)
{
	uint8_t gflag;

	xeWriteByte(dspParamsAddr+p, v);
	if (v == 255) return;
	if (v == 254) {
		xeReadByte(dspGreenParamsAddr+p, &v); // use green param
		gflag = 1;
	} else gflag = 0;
	dspSendCommand(p, v);
	if (p == 62) {SWfilter = 1; SSB_SWfilter = 1;}
	if (p == 63) {SWfilter = 0; SSB_SWfilter = 0;}
//if (p == 61) {SWfilter = 1; CW_SWfilter = 1;}
	if (p == 61) {SWfilter = 1; CW_SWfilter = 1;}	  // ijs Star6a9
	if (p == 64) {SWfilter = 0; CW_SWfilter = 1;}
	if ((p == 6) || (p == 62) || (p == 63)) dspSetSSBfilterCentre(63-SWfilter);
	if ((p == 6) || ((p >= 61) && (p <= 66))) zscShowFilter();
	if (p == 66) {
		if (gflag) {
			xeWriteByte(dspParamsAddr+42, 254);
			xeWriteByte(dspGreenParamsAddr+42, v);
		} else {
			xeWriteByte(dspParamsAddr+42, v);
		}	
	}
	if(p==22){etShowQRPQRO(v);}
}


void dspChangeParamValue(uint8_t starno, uint8_t value) __attribute__((section(".highmem")));
void dspChangeParamValue(uint8_t starno, uint8_t value)
{
	uint8_t max, min, prevvalue;

	max = dspGetMax(starno);
	min = dspGetMin(starno);
	prevvalue = dspGetParamValue(starno);
	if (value > max) value = max;
	if (value < min) value = min;
	if (value == prevvalue) return;
	if (starno == 81) dspStoreTxDrive(value); else dspSetParam(starno, value);
	switch (starno) {
		case 11:
		case 32:
		case 43:
		case 51:
		case 91:
		case 93: zscShowSwitches();
 	}
}  


uint8_t dspGetParamValue(uint8_t starno) __attribute__((section(".highmem")));
uint8_t dspGetParamValue(uint8_t starno)
{
	uint8_t b;

	if (starno == 0) return 255;
	xeReadByte(dspParamsAddr+starno, &b);
	if (b == 254) xeReadByte(dspGreenParamsAddr+starno, &b); // use green param
	return b;
}


uint8_t dspGetParam254Value(uint8_t starno) __attribute__((section(".highmem")));
uint8_t dspGetParam254Value(uint8_t starno)
{
	uint8_t b;

	if (starno == 0) return 255;
	xeReadByte(dspParamsAddr+starno, &b);
	return b;
}


uint8_t dspGetGreenParamValue(uint8_t starno) __attribute__((section(".highmem")));
uint8_t dspGetGreenParamValue(uint8_t starno)
{
	uint8_t b;

 	if (starno == 0) return 255;
	xeReadByte(dspGreenParamsAddr+starno, &b);
	return b;
}


uint8_t dspUserToStar(uint8_t userno) __attribute__((section(".highmem")));
uint8_t dspUserToStar(uint8_t userno)
{
	if ((userno >= 64) && (userno <= 66)) return (userno+3);
	if ((userno >= 73) && (userno <= 78)) return (userno+1);
	if (Mode > 1) {  // ie: CW
		if (userno == 62) return 64;
		if (userno == 63) return 66;
		if (userno == 72) return 73;
		if (userno == 91) return 93;
		if (userno == 92) return 94;
	} else {      //   SSB
		if (userno == 61) return 62;
		if (userno == 62) return 63;
		if (userno == 63) return 65;
		if (userno == 93) return 95;
	}
	return userno;
}


uint8_t dspStarToUser(uint8_t starno) __attribute__((section(".highmem")));
uint8_t dspStarToUser(uint8_t starno)
{
	if ((starno >= 67) && (starno <= 69)) return (starno-3);
	if ((starno >= 74) && (starno <= 79)) return (starno-1);
	if (Mode > 1) {  // ie: CW
		if (starno == 64) return 62;
		if (starno == 66) return 63;
		if (starno == 73) return 72;
		if (starno == 93) return 91;
		if (starno == 94) return 92;
	} else {      //   SSB
		if (starno == 62) return 61;
		if (starno == 63) return 62;
		if (starno == 65) return 63;
		if (starno == 95) return 93;
	}
	return starno;
}


uint8_t dspGetMax(uint8_t starno) __attribute__((section(".highmem")));
uint8_t dspGetMax(uint8_t starno)
{
	uint8_t b;

	xeReadByte(xeGetDspMaxAddr()+starno, &b);
	return b;
}


uint8_t dspGetMin(uint8_t starno) __attribute__((section(".highmem")));
uint8_t dspGetMin(uint8_t starno)
{
	uint8_t b;

	xeReadByte(xeGetDspMinAddr()+starno, &b);
	return b;
}


uint8_t dspGetMult(uint8_t starno) __attribute__((section(".highmem")));
uint8_t dspGetMult(uint8_t starno)
{
	uint8_t b;

	xeReadByte(xeGetDspMultAddr()+starno, &b);
	return b;
}


/////////////////////////////////////////

char* dspParamValString(uint8_t starno, uint8_t LeadingSpaces) __attribute__((section(".highmem")));
char* dspParamValString(uint8_t starno, uint8_t LeadingSpaces)
{
	uint16_t a, n;
	uint8_t d, mult;
	char su[10], sw[10], *pv;

	pv = ParamValueString;	
	ParamVal = dspGetParamValue(starno);
	mult = dspGetMult(starno);
	a = ParamVal * mult;
	n = a / 100;
	d = a % 100;  
	if ((starno == 61) || (starno == 66) || (starno == 42)) n *= 10; //10 cw filter depth in 10dB steps
	ltoa(n, pv, 10);
	if (mult != 100) {
		ltoa(d, sw, 10);
		if (d < 10) {
			strcpy(su,"0");
			strcat(su,sw);
		} else strcpy(su, sw);
		strcat(pv, ".");
		strcat(pv, su);
	}
	n = strlen(pv);
	if ((n < 3) && (LeadingSpaces == 1)) {  // was 4
		strcpy(su, pv);
		if (n == 2) strcpy(pv, " ");
		if (n == 1) strcpy(pv, "  ");
		strcat(pv, su);
	}
	return pv;
}


/////////////////////////////////////////////////

void dspInitGraphicsEA320() __attribute__((section(".highmem")));
void dspInitGraphicsEA320()
{
  dspInitBarsEA320();
	PrevTx = 0;
}




void dspInitMaxMin() __attribute__((section(".highmem")));
void dspInitMaxMin()
{
	uint8_t i;

	for (i = 1; i <= 8; i++) {
		dBmax[i] = 0;
		dBmin[i] = 90; 
	}
}


void dspInitBarsEA320() __attribute__((section(".highmem")));
void dspInitBarsEA320()
{
  uint8_t i,j;
	for(j=0;j<=1;j++)
	{
		for(i=1;i<=8;i++){BarPixEA320[j][i] = 0;};
	}
	PrevAD603pixEA320 = 0;
	PrevSparePosEA320 = 260;  // empty spare gain
	dspInitMaxMin();
}


uint8_t dspCalcDB(uint16_t dspdata) __attribute__((section(".highmem")));
uint8_t dspCalcDB(uint16_t dspdata)
{
  if(dspdata==0){return 0;};
//	x = dspdata;
	y = 8.685889638*(log(32768/dspdata));
	return 90 - round(y);
}


void dspDisplayCalc() __attribute__((section(".highmem")));
void dspDisplayCalc()
{
	Adc = TelemBuf[3]+256*TelemBuf[2];
	HangStuff = TelemBuf[4];
	Dac = TelemBuf[6] + 256*TelemBuf[5];
	uartADCgain = TelemBuf[21] / 16;
	GN = TelemBuf[23];
	hs = HangStuff & 0x3F;
	hff = 63-hs;

  AD603gain = 0;   // = AD603 gain
  uint8_t xg;
	for(xg=0;xg<=41;xg++)
  {
    if(CalCurve[xg] >= hs)
    {
      AD603gain = xg;
      break;
   	}
  }
  Dev1 = TelemBuf[8]  + 256*TelemBuf[7];
  Dev2 = TelemBuf[10] + 256*TelemBuf[9];
  Dev3 = TelemBuf[12] + 256*TelemBuf[11];
  Dev4 = TelemBuf[14] + 256*TelemBuf[13];
  Dev5 = TelemBuf[16] + 256*TelemBuf[15];
  Dev6 = TelemBuf[18] + 256*TelemBuf[17];
	
	Spike = TelemBuf[20];
	Hang = 0;
	if((HangStuff&0x40)>0){Hang=1;};

	if(Transmit==1)
	{
    micgain = GN;
	}
	else
	{
		agcdb = GN-28;
		if(agcdb<0){agcdb=0;};
		if(agcdb>72){agcdb=72;};
		ADCdb = (uartADCgain*150)/100;
		TotalGain = AD603gain + ADCdb + agcdb;
		//  NB hff is DAC value
	}
}


void dspDoBottomDisplay() __attribute__((section(".highmem")));
void dspDoBottomDisplay()
{
	if(DisplaySwitch!=1){return;};
	if(zg!=1){return;};
	DspCount +=1;
	if(DspCount<200){return;};
	DspCount = 0;
///////////////////
	dspDisplayCalc();
	usbCheckPC();
	ctDoCat();
	scCheckFlags();
	if(Transmit==1)
	{
		gedSwitchLayer(LayerTx);      // transmit layer

		strcpy(StrA,"Mic in=");
		ltoa(dspCalcDB(Adc),StrB,10);
		strcat(StrA,StrB);
		strcat(StrA,"dB  ");
		gedF8WriteString(StrA,5,212);


		strcpy(StrA,"After dec=");
		ltoa(dspCalcDB(Dev6),StrB,10);
		strcat(StrA,StrB);
		strcat(StrA,"dB  ");
		gedF8WriteString(StrA,5,221);

		strcpy(StrA,"After Ngate=");
		ltoa(dspCalcDB(Dev1),StrB,10);
		strcat(StrA,StrB);
		strcat(StrA,"dB  ");
		gedF8WriteString(StrA,5,230);

		usbCheckPC();
		ctDoCat();
		scCheckFlags();

		strcpy(StrA,"After filter=");
		ltoa(dspCalcDB(Dev2),StrB,10);
		strcat(StrA,StrB);
		strcat(StrA,"dB  ");
		gedF8WriteString(StrA,97,212);


		strcpy(StrA,"After VOGAD=");
		ltoa(dspCalcDB(Dev4),StrB,10);
		strcat(StrA,StrB);
		strcat(StrA,"dB  ");
		gedF8WriteString(StrA,97,221);

		strcpy(StrA,"After comp=");
		ltoa(dspCalcDB(Dev5),StrB,10);
		strcat(StrA,StrB);
		strcat(StrA,"dB  ");
		gedF8WriteString(StrA,97,230);

		usbCheckPC();
		ctDoCat();
		scCheckFlags();
	
		strcpy(StrA,"Tx mon out");
		ltoa(dspCalcDB(Dev3),StrB,10);
		strcat(StrA,StrB);
		strcat(StrA,"dB  ");
		gedF8WriteString(StrA,194,212);


		strcpy(StrA,"Tx IF out");
		ltoa(dspCalcDB(Dac),StrB,10);
		strcat(StrA,StrB);
		strcat(StrA,"dB  ");
		gedF8WriteString(StrA,194,221);

		strcpy(StrA,"VOGAD gain=");
		ltoa(micgain-64,StrB,10);
		strcat(StrA,StrB);
		strcat(StrA,"dB  ");
		gedF8WriteString(StrA,194,230);

		usbCheckPC();
		ctDoCat();
		scCheckFlags();

	}
	else
	{
		gedSwitchLayer(LayerRx);

		strcpy(StrA,"AD603=");
		ltoa(AD603gain,StrB,10);
		strcat(StrA,StrB);
		strcat(StrA,"dB ");
		gedF8WriteString(StrA,5,212);
		
		strcpy(StrA,"AGC=");
		ltoa(agcdb,StrB,10);
		strcat(StrA,StrB);
		strcat(StrA,"dB  ");
		gedF8WriteString(StrA,5,221);

		strcpy(StrA,"ADC=");
		ltoa(ADCdb,StrB,10);
		strcat(StrA,StrB);
		strcat(StrA,"dB  ");
		gedF8WriteString(StrA,5,230);
	
		usbCheckPC();
		ctDoCat();
		scCheckFlags();
	
		strcpy(StrA,"Rxin=");
		ltoa(dspCalcDB(Adc),StrB,10);
		strcat(StrA,StrB);
		strcat(StrA,"dB  ");
		gedF8WriteString(StrA,68,212);


		strcpy(StrA,"NB=");
		ltoa(dspCalcDB(Dev6),StrB,10);
		strcat(StrA,StrB);
		strcat(StrA,"dB  ");
		gedF8WriteString(StrA,68,221);

		strcpy(StrA,"FLTout=");
		ltoa(dspCalcDB(Dev1),StrB,10);
		strcat(StrA,StrB);
		strcat(StrA,"dB  ");
		gedF8WriteString(StrA,68,230);

		usbCheckPC();
		ctDoCat();
		scCheckFlags();

		strcpy(StrA,"AGCout=");
		ltoa(dspCalcDB(Dev2),StrB,10);
		strcat(StrA,StrB);
		strcat(StrA,"dB  ");
		gedF8WriteString(StrA,140,212);


		strcpy(StrA,"ANout=");
		ltoa(dspCalcDB(Dev4),StrB,10);
		strcat(StrA,StrB);
		strcat(StrA,"dB  ");
		gedF8WriteString(StrA,140,221);

		strcpy(StrA,"N/Rout=");
		ltoa(dspCalcDB(Dev5),StrB,10);
		strcat(StrA,StrB);
		strcat(StrA,"dB  ");
		gedF8WriteString(StrA,140,230);

		usbCheckPC();
		ctDoCat();
		scCheckFlags();
	
		strcpy(StrA,"preAF=");
		ltoa(dspCalcDB(Dev3),StrB,10);
		strcat(StrA,StrB);
		strcat(StrA,"dB  ");
		gedF8WriteString(StrA,211,212);


		strcpy(StrA,"AFout=");
		ltoa(dspCalcDB(Dac),StrB,10);
		strcat(StrA,StrB);
		strcat(StrA,"dB  ");
		gedF8WriteString(StrA,211,221);

		strcpy(StrA,"TotGN=");
		ltoa(TotalGain,StrB,10);
		strcat(StrA,StrB);
		strcat(StrA,"dB  ");
		gedF8WriteString(StrA,211,230);

	}
	gedSwitchLayer(LayerTR);              // TR layer
}


void dspEA320eightBars() __attribute__((section(".highmem")));
void dspEA320eightBars()
{
  uint8_t i;
  uint8_t Yi;
	uint16_t dbpix;
	uint16_t db;
	uint16_t pw;
	for(i=1;i<=8;i++)
	{
		switch(i)
		{
			case 1: dspdata = Adc;  break;
			case 2: dspdata = Dev6; break;
			case 3: dspdata = Dev1; break;
			case 4: dspdata = Dev2; break;
			case 5: dspdata = Dev4; break;
			case 6: dspdata = Dev5; break;
			case 7: dspdata = Dev3; break;
			case 8: dspdata = Dac;  break;
		}
		db = dspCalcDB(dspdata);
		if(db>90){db=90;};
		if(db>dBmax[i]){dBmax[i]=db;};
		if(db<dBmin[i]){dBmin[i]=db;};
		
		
		dbpix = (db*160)/100;
		pw = BarPixEA320[Transmit][i];
		BarPixEA320[Transmit][i] = dbpix;
		Yi = BarsY +11*(i-1);
		ltoa(db,StrA,10);
		gedF8WriteString(StrA,70,Yi+2);		
		ltoa(dBmin[i],StrA,10);
		gedF8WriteString(StrA,245,Yi+2);		
		ltoa(dBmax[i],StrA,10);
		gedF8WriteString(StrA,265,Yi+2);		
		if(dbpix!=pw)
		{
			if(dbpix>pw)
			{
				Brush.Color = 1;
				gedFillRect(84+pw,Yi+3,84+dbpix,Yi+8);
			}
			else
			{
				Brush.Color = 0;
				gedFillRect(84+dbpix+1,Yi+3,84+pw,Yi+8);	
			}
		}
		Brush.Color = 1;
		usbCheckPC();
		ctDoCat();
		scCheckFlags();
	}
}
		

void dspEA320GainBar() __attribute__((section(".highmem")));
void dspEA320GainBar()
{
	uint8_t i,np, SpareGain;
	uint16_t xp;   //  will be muliplied by 2
	AD603pixEA320 = (AD603gain*150)/100;
//	if(ADCdb<1){ADCdb=1;};
	ADCpixEA320 = (ADCdb*150)/100;
	AGCpixEA320 = (agcdb*150)/100;
	if(AD603pixEA320 > PrevAD603pixEA320)
	{
		Brush.Color = 1;
		gedFillRect(72+PrevAD603pixEA320,101,72+AD603pixEA320,109);  // AD603 bar  7 pix solid
	}
	else
	{
		Brush.Color = 0;
		gedFillRect(72+AD603pixEA320+1,101,72+PrevAD603pixEA320,109);	
	}
	PrevAD603pixEA320 = AD603pixEA320;
	if(ADCpixEA320>0)
	{
		Brush.Color=0;
		gedFillRect(72+AD603pixEA320+1,101,72+AD603pixEA320+ADCpixEA320,109);  // ADC bar
		Brush.Color=1;
		xp=72+AD603pixEA320+1;
		np=ADCpixEA320/2;
		for(i=0;i<=np;i++)
		{
			gedFillRect(xp+2*i+1,101,xp+2*i+1,109); // striped ADC bar
		}	 
	}
	if(AGCpixEA320>0)
	{
		Brush.Color=0;
		gedFillRect(72+AD603pixEA320+ADCpixEA320+1,101,72+AD603pixEA320+ADCpixEA320+AGCpixEA320,109);  // clear
		Brush.Color = 1;
		gedFillRect(72+AD603pixEA320+ADCpixEA320+1,104,72+AD603pixEA320+ADCpixEA320+AGCpixEA320,109);  // AGC bar
	}
	Brush.Color=0;
	xp = 72+AD603pixEA320+ADCpixEA320+AGCpixEA320+1;  // start of spare gain
	xp = (xp/2)*2;  // keep the vertical stripes on even pixel positions!
	if(xp<PrevSparePosEA320)
	{
		np = (PrevSparePosEA320-xp-3)/2 + 1;
		gedFillRect(xp,101,PrevSparePosEA320,109);  // clear spare gain
		Brush.Color=1;
		for(i=0;i<=np;i++)
		{
			gedFillRect(xp+2*i+1,104,xp+2*i+1,109);
		}	
	}
	PrevSparePosEA320 = xp;
	usbCheckPC();
	ctDoCat();
	scCheckFlags();
	
	strcpy(StrA,PMS(s_Gain));
	gedF8WriteString(StrA,72,80);
	ltoa(TotalGain,StrA,10);
	strcat(StrA,"dB");
	if(TotalGain<100){strcat(StrA,pmdSpaces(2));};
	gedF8WriteString(StrA,72,90);
	
	SpareGain = 123-TotalGain;
	ltoa(SpareGain,StrB,10);  // spare gain
	Brush.Color=0;
	gedFillRect(263,100,279,108);
	Brush.Color=1;
	StrA[0]=0;
	if(SpareGain<100){strcpy(StrA," ");};
	strcat(StrA,StrB);
	gedF8WriteString(StrA,263,100);
	strcpy(StrA,PMS(s_dB));
	gedF8WriteString(StrA,267,109);

	strcpy(StrA,PMS(s_DAC));
	gedF8WriteString(StrA,70,118);
	ltoa(hff,StrA,10);
	strcat(StrA,pmdSpaces(2));
	gedF8WriteString(StrA,70,127);

	strcpy(StrA,PMS(s_AD603));
	gedF8WriteString(StrA,94,118);
	ltoa(AD603gain,StrA,10);
	strcat(StrA,PMS(s_dB__));
	gedF8WriteString(StrA,97,127);

	strcpy(StrA,PMS(s_ADC));
	gedF8WriteString(StrA,132,118);
	ltoa(ADCdb,StrA,10);
	strcat(StrA,PMS(s_dB__));
	gedF8WriteString(StrA,130,127);

	strcpy(StrA,PMS(s_AGC));
	gedF8WriteString(StrA,172,118);
	ltoa(agcdb,StrA,10);
	strcat(StrA,PMS(s_dB__));
	gedF8WriteString(StrA,170,127);

	if(Hang==1){strcpy(StrA,PMS(s_Hang));}else{strcpy(StrA,"      ");};
	gedF8WriteString(StrA,245,119);

	if(Spike==1){strcpy(StrA,PMS(s_Spike));}else{strcpy(StrA,"      ");};
	gedF8WriteString(StrA,245,129);	 
}


void dspEA320VOGADgain() __attribute__((section(".highmem")));
void dspEA320VOGADgain()
{
	ltoa(micgain-64,StrB,10);
	strcpy(StrA,PMS(s_VOGAD_gain_));
	strcat(StrA,StrB);
	strcat(StrA,PMS(s_dB__));	
	gedF8WriteString(StrA,72,128);
}


////////////    DSP monitor - mainly in display specific modules

void dspDoDspDisplay() __attribute__((section(".highmem")));
void dspDoDspDisplay()
{
	// note - that if zt (colour TFT present) DisplaySwitch is never set to 1
	if((DisplaySwitch!=1)&&(DisplaySwitch!=2)){return;};
	if ((zg) && (DisplaySwitch == 1))
	{
		dspDoBottomDisplay();
		return;
	}
	if((HaltDSPdisplay==1)||(DisplaySwitch!=2)){return;};
	if ((zt) || (zg))
	{
		if(Transmit!=PrevTx)
		{
	  	dspInitMaxMin();
			PrevTx = Transmit;
			DspCount=100;
		}
		DspCount +=1;
		if(DspCount<100){return;};
		DspCount = 0;
	///////////////////
		dspDisplayCalc();
	
		usbCheckPC();
		ctDoCat();
		scCheckFlags();
		if(zt)
		{
			if(Transmit==1)
			{
				dspEATFTeightBars();
				dspEATFTVOGADgain();
			}
			else
			{
				dspEATFTeightBars();
				dspEATFTGainBar();
			}		
		}
		if ((zg) && (!zt))  // no dual DSP monitor   colour TFT wins
		{
			if(Transmit==1)
			{
				gedSwitchLayer(LayerTx);      // transmit layer
				dspEA320eightBars();
				dspEA320VOGADgain();
			}
			else
			{
				gedSwitchLayer(LayerRx);
				dspEA320eightBars();
				dspEA320GainBar();
			}
			gedSwitchLayer(LayerTR);
		}
	}
}


/////////////////////    EATFT ////////////////////


void dspStartDSPdisplayEATFT() __attribute__((section(".highmem")));
void dspStartDSPdisplayEATFT()
{
	if(Transmit==1){

   	etDSPmonTx();
				   }

	else{
	
if(rx2 == 1 && SWdenoise==1 && SWautonotch == 1){
	  dspSendCommand(159,1);   //hfr per SNRL
    _delay_ms(90);           //  wait for parameter to settle
	  etDSPmonRx2();   
	
	}
else {   dspSendCommand(159,0);

         etDSPmonRx();}

	}

	HaltDSPdisplay = 0;
  dspInitBarsEATFT();
	PrevTx = 0;
}



void dspInitBarsEATFT() __attribute__((section(".highmem")));
void dspInitBarsEATFT()
{
  uint8_t i,j;
	for(j=0;j<=1;j++)
	{
		for(i=1;i<=8;i++){BarPixEATFT[j][i] = 0;};
	}
	PrevAD603pixEATFT = 0;
	PrevSparePosEATFT = 378;  // empty spare gain
	PrevAGCpixEATFT = 0;
	dspInitMaxMin();
}



void dspEATFTeightBars() __attribute__((section(".highmem")));
void dspEATFTeightBars()
{
	uint8_t i , b =0;
	uint16_t Yi,Di =0;
	uint16_t dbpix;
	uint16_t db;
	uint16_t pw;

	if(rx2 == 1) {Di = (Dev6*10);}
	 else {Dev6 = Dev6;} //hfr
	for (i = 1; i <= 8; i++) {
		switch (i) {
			case 1: dspdata = Adc;  break;
			case 2: dspdata = Di; break;  //Dev6
			case 3: dspdata = Dev1; break;
			case 4: dspdata = Dev2; break;
			case 5: dspdata = Dev4; break;
			case 6: dspdata = Dev5; break;
			case 7: dspdata = Dev3; break;
			case 8: dspdata = Dac;  break;
		}
		db = dspCalcDB(dspdata);
		if(rx2 == 1 && i == 3 && SWdenoise==1 && SWautonotch == 1 )
	    {db = (Dev1 + 1);
		b = 3; 
		}
		else {
		b = 7;
				} //hfr 
		if(db>90){db=90;};
		if(db>dBmax[i]){dBmax[i]=db;};
		if(db<dBmin[i]){dBmin[i]=db;};

	
		
		etcFont(21);   // Arial12 with 9 pixel height
		dbpix = db*2;
		pw = BarPixEATFT[Transmit][i];
		BarPixEATFT[Transmit][i] = dbpix;
		if(rx2 == 1 && i == 3 && SWdenoise==1 && SWautonotch == 1 ){dbpix = 0;}
		Yi = 163 + (12 * i);
		ltoa(db,StrA,10); //AVR Lib routine 
		if (StrA[1] == 0) {StrA[1] = StrA[0]; StrA[0] = ' '; StrA[2] = ' '; StrA[3] = 0;}
		etcTextColour(b,1);
//		etcColourRect(148,Yi+1,148+18,Yi+11,1);
		etcString(157,Yi+2,'C',StrA);			// ddB
		ltoa(dBmin[i],StrA,10);
		if (StrA[1] == 0) {StrA[1] = StrA[0]; StrA[0] = ' '; StrA[2] = ' '; StrA[3] = 0;}
		etcTextColour(14,1);
//		etcColourRect(356,Yi+1,356+21,Yi+11,1);
		etcString(367,Yi+2,'C',StrA);			// min
		ltoa(dBmax[i],StrA,10);
		if (StrA[1] == 0) {StrA[1] = StrA[0]; StrA[0] = ' '; StrA[2] = ' '; StrA[3] = 0;}
//		etcColourRect(379,Yi+1,381+19,Yi+11,1);
		etcString(390,Yi+2,'C',StrA);			//max		
		ltoa(dBmax[i]-dBmin[i],StrA,10);
		if (StrA[1] == 0) {StrA[1] = StrA[0]; StrA[0] = ' '; StrA[2] = ' '; StrA[3] = 0;}
		etcTextColour(10,1);
//		etcColourRect(402,Yi+1,403+18,Yi+11,1);
		etcString(411,Yi+2,'C',StrA);			// diff		
		etcFont(19);   // Arial12

		if(dbpix!=pw)
		{
			if(dbpix>pw)
			{
				etcDisplayColour(10,1);
				etcFillRect(168+pw,Yi+3,168+dbpix,Yi+9);
			}
			else
			{
				etcDisplayColour(1,1);
				etcFillRect(168+dbpix+1,Yi+3,168+pw,Yi+9);	
			}
		}

		usbCheckPC();
		ctDoCat();
		scCheckFlags();
	}
}
		

void dspEATFTGainBar() __attribute__((section(".highmem")));
void dspEATFTGainBar()
{
	uint16_t GainX=129;
	uint16_t GainY=119;
	uint16_t SpareGain;
	uint16_t xp;   //  will be muliplied by 2
	AD603pixEATFT = AD603gain*2;
	ADCpixEATFT = ADCdb*2;
	AGCpixEATFT = agcdb*2;
	if(AD603pixEATFT > PrevAD603pixEATFT)
	{
		etcDisplayColour(15,1);
		etcFillRect(GainX+PrevAD603pixEATFT,GainY,GainX+AD603pixEATFT,GainY+8);  // AD603 bar  7 pix solid
	}
	else
	{
		etcDisplayColour(1,1);
		etcFillRect(GainX+AD603pixEATFT,GainY,GainX+PrevAD603pixEATFT,GainY+8);	
	}
	PrevAD603pixEATFT = AD603pixEATFT;

	if(ADCpixEATFT>0)
	{
		etcDisplayColour(12,1);//(1,1);
 		etcFillRect(GainX+AD603pixEATFT,GainY,GainX+AD603pixEATFT+ADCpixEATFT,GainY+8);  // ADC bar
	}

	if(AGCpixEATFT > PrevAGCpixEATFT)
	{
		etcDisplayColour(11,1);
		etcFillRect(GainX+AD603pixEATFT+ADCpixEATFT+1,GainY,GainX+AD603pixEATFT+ADCpixEATFT+AGCpixEATFT,GainY+8);  // AGC bar
	}
	PrevAGCpixEATFT = AGCpixEATFT;


//	etcDisplayColour(1,1);
	xp = GainX+AD603pixEATFT+ADCpixEATFT+AGCpixEATFT+1;  // start of spare gain
	if(xp<PrevSparePosEATFT)
	{
		etcDisplayColour(18,1);
		etcFillRect(xp,GainY,PrevSparePosEATFT,GainY+8);
	}
	PrevSparePosEATFT = xp;
	usbCheckPC();
	ctDoCat();
	scCheckFlags();
	
	etcTextColour(7,1);
	ltoa(TotalGain,StrA,10);
	strcat(StrA,"dB   ");
	if(TotalGain<100){strcat(StrA,pmdSpaces(2));};
	etcString(57+32,118,'L',StrA);
	
	SpareGain = 123-TotalGain;
	ltoa(SpareGain,StrB,10);  // spare gain
	etcTextColour(18,1);
	StrA[0]=0;
	if(SpareGain<100){strcpy(StrA," ");};
	strcat(StrA,StrB);
	strcat(StrA,PMS(s_dB__));
	etcString(57+325,118,'L',StrA);

	ltoa(hff,StrA,10);
	etcTextColour(10,1);
	etcString(88,144,'L',StrA);

	ltoa(AD603gain,StrA,10);
	strcat(StrA,PMS(s_dB__));
	etcTextColour(15,1);
	etcString(153,144,'L',StrA);

	ltoa(ADCdb,StrA,10);
	strcat(StrA,PMS(s_dB__));
	etcTextColour(12,1);
	etcString(200,144,'L',StrA);

	ltoa(agcdb,StrA,10);
	strcat(StrA,PMS(s_dB__));
	etcTextColour(11,1);
	etcString(250,144,'L',StrA);

	etcTextColour(12,1);
	if(Hang==1){strcpy(StrA,PMS(s_Hang));}else{strcpy(StrA,pmdSpaces(9));};
	etcString(320,137,'L',StrA);

	etcTextColour(13,1);
	if(Spike==1){strcpy(StrA,PMS(s_Spike));}else{strcpy(StrA,pmdSpaces(9));};
	etcString(365,137,'L',StrA);	 
}


void dspEATFTVOGADgain() __attribute__((section(".highmem")));
void dspEATFTVOGADgain()
{
	ltoa(micgain-64,StrA,10);
	strcat(StrA,PMS(s_dB__));	
	etcTextColour(12,1);
	etcString(140,130,'L',StrA);
}
