/*
 *****    homebrew-radios.net   *****

 *****     Project TxrAVR       *****

 taKS0108Driver.c    128x64 mono graphics driver routines

*/

//***********************************************************************

//   // See UC0180 datasheet SETY is horzontal and SETX is vertical !!!

//***********************************************************************


#include "taGlobal.h"
#include "taKS0108Driver.h"
#include "taFT245R.h"
#include "taFont8.h"
#include "taFont16.h"

#define KS_DATA_PORT	PORTC
#define KS_DATA_DIR		DDRC
#define KS_DATA_PIN		PINC

#define KS_RS_PORT		PORTD
#define KS_RW_PORT		PORTG
#define KS_EN_PORT		PORTG
#define KS_RST_PORT		PORTL
#define KS_CS1_PORT		PORTL
#define KS_CS2_PORT		PORTL

#define KS_RS			(1 << 7)
#define KS_RW			(1 << 0)
#define KS_EN			(1 << 1)
#define KS_RST		(1 << 6)

#define KS_CS1		(1 << 7)
#define KS_CS2		(1 << 4)

#define KS_WIDTH		128
#define KS_HEIGHT	64

#define KS_YSET       	0x40
#define KS_XSET       	0xB8
#define KS_START_LINE 	0xC0
#define KS_ON						0x3F
#define KS_OFF					0x3E
#define KS_BUSY	0x80

#define ksdLetterSpacing 1
#define ksdLineSpacing 9

static uint8_t screen_x;
static uint8_t screen_y;
static uint16_t ksdNextCharX;
static uint8_t DisplayEnabled;

uint8_t ksColour;

void ksdXBACK();
void ksdDoByte(uint8_t X,uint8_t Y, uint8_t mask);

void ksdEnableController(uint8_t controller) __attribute__((section(".highmem")));
void ksdEnableController(uint8_t controller)
{
	switch (controller)
	{
		case 0 : if(ksCSLOW) KS_CS1_PORT &= ~KS_CS1; else KS_CS1_PORT |= KS_CS1; break;
		case 1 : if(ksCSLOW) KS_CS2_PORT &= ~KS_CS2; else KS_CS2_PORT |= KS_CS2; break;
	}
}


void ksdDisableController(uint8_t controller) __attribute__((section(".highmem")));
void ksdDisableController(uint8_t controller)
{
	switch (controller)
	{
		case 0 : if(ksCSLOW) KS_CS1_PORT |= KS_CS1; else KS_CS1_PORT &= ~KS_CS1; break;
		case 1 : if(ksCSLOW) KS_CS2_PORT |= KS_CS2; else KS_CS2_PORT &= ~KS_CS2; break;
	}
}


uint8_t ksdReadStatus(uint8_t controller) __attribute__((section(".highmem")));
uint8_t ksdReadStatus(uint8_t controller)
{
	uint8_t status;
	
	if (!DisplayEnabled) return 0;
	usbINTenable(3, 0); // disable Rx interrupt INT3
	KS_DATA_DIR = 0x00;
	KS_RW_PORT |= KS_RW;
	KS_RS_PORT &= ~KS_RS;
	ksdEnableController(controller);
	KS_EN_PORT |= KS_EN;
	_delay_us(1);
	status = KS_DATA_PIN;
	KS_EN_PORT &= ~KS_EN;
	ksdDisableController(controller);
	if (RxIntEnabled == 1) usbINTenable(3, 1); // enable Rx interrupt INT3
	return status;
}


void ksdWriteCommand(uint8_t commandToWrite, uint8_t controller) __attribute__((section(".highmem")));
void ksdWriteCommand(uint8_t commandToWrite, uint8_t controller)
{
	if (!DisplayEnabled) return;
	while (ksdReadStatus(controller) & KS_BUSY);
	usbINTenable(3, 0); // disable Rx interrupt INT3
	KS_DATA_DIR = 0xFF;
	KS_RW_PORT &= ~(KS_RW | KS_RS);
	ksdEnableController(controller);
	KS_DATA_PORT = commandToWrite;
	KS_EN_PORT |= KS_EN;
	_delay_us(1);
	KS_EN_PORT &= ~KS_EN;
	ksdDisableController(controller);
	if (RxIntEnabled == 1) usbINTenable(3, 1); // enable Rx interrupt INT3
}


uint8_t ksdReadData(void) __attribute__((section(".highmem")));
uint8_t ksdReadData(void)
{
	uint8_t data;
	if(screen_x > 127){return 0;};
	if (!DisplayEnabled) return 0;
	while (ksdReadStatus(screen_x / 64) & KS_BUSY);
	usbINTenable(3, 0); // disable Rx interrupt INT3
	KS_DATA_DIR = 0x00;
	KS_RS_PORT |= (KS_RW | KS_RS);
	ksdEnableController(screen_x / 64);
	KS_EN_PORT |= KS_EN;
	_delay_us(1);
	data = KS_DATA_PIN;
	KS_EN_PORT &= ~KS_EN;
	ksdDisableController(screen_x / 64);
	if (RxIntEnabled == 1) usbINTenable(3, 1); // enable Rx interrupt INT3
	screen_x++;
	return data;
}


void ksdWriteData(uint8_t dataToWrite) __attribute__((section(".highmem")));
void ksdWriteData(uint8_t dataToWrite)
{
	if(screen_x > 127){return;};
	if (!DisplayEnabled) return;
	while(ksdReadStatus(screen_x / 64) & KS_BUSY);
	usbINTenable(3, 0); // disable Rx interrupt INT3
	KS_DATA_DIR = 0xFF;
	KS_RW_PORT &= ~KS_RW;
	KS_RS_PORT |= KS_RS;
	KS_DATA_PORT = dataToWrite;
	ksdEnableController(screen_x / 64);
	KS_EN_PORT |= KS_EN;
	_delay_us(1);
	KS_EN_PORT &= ~KS_EN;
	ksdDisableController(screen_x / 64);
	if (RxIntEnabled == 1) usbINTenable(3, 1); // enable Rx interrupt INT3
	screen_x++;
}


void ksdInitalizePorts(void) __attribute__((section(".highmem")));
void ksdInitalizePorts(void)
{
	KS_CS1_PORT |= KS_CS1;
	KS_CS2_PORT |= KS_CS2;
	KS_RST_PORT |= KS_RST;
}


uint8_t ksdReadPROGMEMbyte(char * ptr) __attribute__((section(".highmem")));
uint8_t ksdReadPROGMEMbyte(char * ptr)
{
	return pgm_read_byte(ptr);
}


void ksdInit(void) __attribute__((section(".highmem")));
void ksdInit(void)
{
	ksColour = 1;
	uint8_t i;
	
	if (!zs) {DisplayEnabled = 0; return;} else DisplayEnabled = 1;
	ksdInitalizePorts();
	
	// Ensure 128x64 display is connected and operating
	for (i = 0; i < 200; i++) {if (ksdReadStatus(0) & KS_BUSY) _delay_ms(1); else break;}
	if (i >= 200) {zs = 0; DisplayEnabled = 0; return;}
	for (i = 0; i < 200; i++) {if (ksdReadStatus(1) & KS_BUSY) _delay_ms(1); else break;}
	if (i >= 200) {zs = 0; DisplayEnabled = 0; return;}

	for (i = 0; i < 2; i++)
	{ 
		ksdWriteCommand((KS_ON), i);
	  	ksdWriteCommand(KS_START_LINE | 0, i);
	}
}


// See UC0180 datasheet SETY is horzontal and SETX is vertical !!!
void ksdGoTo(uint8_t x, uint8_t y, uint8_t reading) __attribute__((section(".highmem")));
void ksdGoTo(uint8_t x, uint8_t y, uint8_t reading)
{
	screen_y = y;
 	uint8_t side = x/64;
	uint8_t otherside;
	if (side == 1) otherside = 0; else otherside = 1;
	ksdWriteCommand(KS_XSET | y, otherside);
	ksdWriteCommand(KS_YSET | 0, otherside); //if on left side then needs to flow into left edge 
	if (reading == 1)
	{
	 	screen_x = 64*otherside;
		ksdReadData();
	}
	ksdWriteCommand(KS_XSET | y, side);
	ksdWriteCommand(KS_YSET | x%64, side);     // of right side on screen)x ++
	screen_x = x;
 	if (reading == 1)ksdReadData();
	screen_x = x;
}


void ksdXback() __attribute__((section(".highmem")));
void ksdXback()
{
	uint8_t x = --screen_x;
 	uint8_t side = x/64;
	ksdWriteCommand(KS_YSET | x%64, side);
}


void ksdClearScreen(void) __attribute__((section(".highmem")));
void ksdClearScreen(void)
{
	uint8_t i, j;
	
	for (j = 0; j < 8; j++)
	{
  		ksdGoTo(0, j, 0);
  		for (i = 0; i < KS_WIDTH; i++)
    	{
			ksdWriteData(0x00);
	  	}
	}
}


void ksdSetPixel(uint8_t x, uint8_t y, uint8_t color) __attribute__((section(".highmem")));
void ksdSetPixel(uint8_t x, uint8_t y, uint8_t color)
{
	uint8_t tmp;	
	ksdGoTo(x, (y / 8), 0);
	tmp = ksdReadData();
	ksdGoTo(x, (y / 8), 0);
	tmp = ksdReadData();
	ksdGoTo(x, (y / 8), 0);
	if(color==1)
	{tmp |= (1 << (y % 8));}
	else
	{tmp &= ~(1 << (y % 8));}
	ksdWriteData(tmp);
}


void ksdBitmap(char * bmp, uint8_t x, uint8_t y, uint8_t dx, uint8_t dy) __attribute__((section(".highmem")));
void ksdBitmap(char * bmp, uint8_t x, uint8_t y, uint8_t dx, uint8_t dy)
{
	uint8_t i, j;
	
	for (j = 0; j < dy / 8; j++)
	{
		ksdGoTo(x, y + j, 0);
		for (i = 0; i < dx; i++) ksdWriteData(ksdReadPROGMEMbyte(bmp++));
	}
}


void ksdRectangle(uint8_t x, uint8_t y, uint8_t b, uint8_t a) __attribute__((section(".highmem")));
void ksdRectangle(uint8_t x, uint8_t y, uint8_t b, uint8_t a)
{
	uint8_t j;
	
	for (j = 0; j < a; j++)
	{
		ksdSetPixel(x, y + j, ksColour);
		ksdSetPixel(x + b - 1, y + j, ksColour);
	}
	for (j = 0; j < b; j++)
	{
		ksdSetPixel(x + j, y, ksColour);
		ksdSetPixel(x + j, y + a - 1, ksColour);
	}
}


void ksdCircle(uint8_t cx, uint8_t cy, uint8_t radius) __attribute__((section(".highmem")));
void ksdCircle(uint8_t cx, uint8_t cy, uint8_t radius)
{
	int x, y, xchange, ychange, radiusError;
	x = radius;
	y = 0;
	xchange = 1 - 2 * radius;
	ychange = 1;
	radiusError = 0;
	while (x >= y)
	{
  		ksdSetPixel(cx+x, cy+y, ksColour); 
  		ksdSetPixel(cx-x, cy+y, ksColour); 
  		ksdSetPixel(cx-x, cy-y, ksColour);
  		ksdSetPixel(cx+x, cy-y, ksColour); 
  		ksdSetPixel(cx+y, cy+x, ksColour); 
  		ksdSetPixel(cx-y, cy+x, ksColour); 
  		ksdSetPixel(cx-y, cy-x, ksColour); 
  		ksdSetPixel(cx+y, cy-x, ksColour); 
  		y++;
  		radiusError += ychange;
  		ychange += 2;
  		if ((2*radiusError + xchange) > 0)
    	{
    		x--;
				radiusError += xchange;
				xchange += 2;
		}
	}
}


void ksdLine(uint16_t X1, uint16_t Y1, uint16_t X2, uint16_t Y2) __attribute__((section(".highmem")));
void ksdLine(uint16_t X1, uint16_t Y1, uint16_t X2, uint16_t Y2)
{
	int CurrentX, CurrentY, Xinc, Yinc; 
	int  Dx, Dy, TwoDx, TwoDy; 
	int TwoDxAccumulatedError, TwoDyAccumulatedError;
	Dx = (X2-X1);
	Dy = (Y2-Y1);
	TwoDx = Dx + Dx;
	TwoDy = Dy + Dy;
	CurrentX = X1;
	CurrentY = Y1;
	Xinc = 1;
	Yinc = 1; 

	if (Dx < 0)
	{
		Xinc = -1;
		Dx = -Dx; 
		TwoDx = -TwoDx; 
	}

	if (Dy < 0)
	{
		Yinc = -1;
		Dy = -Dy; 
		TwoDy = -TwoDy;
	}

	ksdSetPixel(X1, Y1, ksColour);

	if ((Dx != 0) || (Dy != 0))
	{
		if (Dy <= Dx)
		{
			TwoDxAccumulatedError = 0;
			do
			{
				CurrentX += Xinc; 
				TwoDxAccumulatedError += TwoDy;
				if (TwoDxAccumulatedError > Dx) 
				{
					CurrentY += Yinc; 
					TwoDxAccumulatedError -= TwoDx;
				}
				ksdSetPixel(CurrentX, CurrentY, ksColour);
			} while (CurrentX != X2); 
		}
		else 
		{
			TwoDyAccumulatedError = 0; 
			do
			{
				CurrentY += Yinc; 
				TwoDyAccumulatedError += TwoDx;
				if (TwoDyAccumulatedError > Dy) 
				{
					CurrentX += Xinc;
					TwoDyAccumulatedError -= TwoDy;
				}
				ksdSetPixel(CurrentX,CurrentY, ksColour); 
			} while (CurrentY != Y2);
		}
	}
}

void ksdDoByte(uint8_t X,uint8_t Y, uint8_t mask) __attribute__((section(".highmem")));
void ksdDoByte(uint8_t X,uint8_t Y, uint8_t mask)
{
	ksdGoTo(X,Y,1);
	uint8_t b = ksdReadData();
	if(ksColour==1){b |= mask;}else{b &= (~mask);};
	ksdGoTo(X,Y,0);
	ksdWriteData(b);
}
  
void ksdFillRect(uint8_t X1, uint8_t Y1, uint8_t X2, uint8_t Y2) __attribute__((section(".highmem")));
void ksdFillRect(uint8_t X1, uint8_t Y1, uint8_t X2, uint8_t Y2)
{
	uint8_t x,y;
	if((X1>X2)||(Y1>Y2)){return;};
	if((X2>127)||(Y2>63)){return;};
	uint8_t yb1 = Y1 / 8;
	uint8_t yb2 = Y2 / 8; 	
	uint8_t topbits = (0xFF << (Y1 % 8));
	uint8_t botbits = (0xFF >> (7 - (Y2 % 8))); 
	if(yb1==yb2){topbits &= botbits;};
	for(x=X1;x<=X2;x++){ksdDoByte(x,yb1,topbits);}
	if(yb1==yb2){return;};
	if((yb2-yb1)>1)
	{
		for(y=yb1+1;y<yb2;y++)
		{
			for(x=X1;x<=X2;x++){ksdDoByte(x,y,0xFF);}
		}	
	}
	for(x=X1;x<=X2;x++){ksdDoByte(x,yb2,botbits);}
}


void ksdClearRect(uint8_t X1, uint8_t Y1, uint8_t X2, uint8_t Y2) __attribute__((section(".highmem")));
void ksdClearRect(uint8_t X1, uint8_t Y1, uint8_t X2, uint8_t Y2)
{
	ksColour = 0;
	ksdFillRect(X1,Y1,X2,Y2);
	ksColour = 1;
}


void ksdF8WriteChar(uint8_t ch, uint16_t* ppX, uint16_t* ppY, uint8_t inverted) __attribute__((section(".highmem")));
void ksdF8WriteChar(uint8_t ch, uint16_t* ppX, uint16_t* ppY, uint8_t inverted)
{
	uint8_t x, cbits,sbits;
	uint8_t a, b, Y;
	uint8_t mask;
	uint8_t cbyte[12];
	uint8_t bits[12];
	uint8_t * pDigitWords = (uint8_t*)pgm_read_word(&chrtbl_S[ch-32]);
	uint8_t chlen = pgm_read_byte(&lentbl_S[ch-32]) + ksdLetterSpacing;
	Y = *ppY / 8; // upper byte y
	a = *ppY % 8;
	b = 8 - a;
	if((*ppX + chlen -1) > 127) return;  // will crash past screeen right
	mask = (0xFF >> b);	
// if inverted we want an extra black row of pixels above the character
// so if character doesn't align with byte this extra row is in the top byte
// of the character - so simply alter the mask
	ksdGoTo(*ppX, Y, 1);
	for (x = 0; x < chlen; x++)
	{
		if (x < (chlen-ksdLetterSpacing)) cbyte[x] = pgm_read_byte(&pDigitWords[x]);
		else cbyte[x] = 0;
		cbits = (cbyte[x] << a);
		sbits = ksdReadData() & mask;
		if (inverted)
		{
			cbits ^= (~mask); 
			if(a!=0) cbits |= (1 << (a-1));	
		}
		bits[x] = sbits | cbits;
// if inverted and character is screen byte aligned  (ie a==0)	
// then we have to add the extra black row if pixels to the byte above
// but not if we are at the top of the screen (ie: *ppY = 0 givining Y==0 and a==0)	
		if((inverted)&&(a==0)&&(Y>0))
		{
			ksdGoTo(*ppX + x,Y-1,1);
			sbits = ksdReadData() | 0x80;
			ksdGoTo(*ppX + x,Y-1,0);
		 	ksdWriteData(sbits);
			ksdGoTo(*ppX + x + 1,Y,1); // we have have killed the auto increment sequntial reading !!
		}
	}
	ksdGoTo(*ppX, Y, 0);
	for (x = 0; x < chlen; x++)
	{ 
	 	ksdWriteData(bits[x]);
	}

	mask = (0xFF << a);		
	ksdGoTo(*ppX, Y+1, 1);
	for (x = 0; x < chlen; x++)
	{
		cbits = (cbyte[x] >> b);
		sbits = ksdReadData() & mask;
		if (inverted) cbits ^= (~mask); 
		bits[x] = sbits | cbits;
	}
	ksdGoTo(*ppX, Y+1, 0);
	for (x = 0;x < chlen; x++)
	{ 
	 	ksdWriteData(bits[x]);
	}
	*ppX += chlen;  // position for next (proportional spacing)

}


void ksdF8WriteString(char* s, uint16_t X, uint16_t Y, uint8_t inverted) __attribute__((section(".highmem")));
void ksdF8WriteString(char* s, uint16_t X, uint16_t Y, uint8_t inverted)
{
	uint16_t StartX = X;
	uint16_t nc = strlen(s);
	char ch;
	for (uint16_t i = 0; i < nc; i++)
	{
		ch = s[i];
		if (ch != 10)
		{
			if (ch == 13)
			{
				Y += ksdLineSpacing;
				X = StartX;
			}
			else {ksdF8WriteChar(s[i], &X, &Y, inverted);}
		}
	}
	ksdNextCharX = X;
}

void ksdF8CentreWriteString(char* s, uint16_t X, uint16_t Y, uint8_t inverted) __attribute__((section(".highmem")));
void ksdF8CentreWriteString(char* s, uint16_t X, uint16_t Y, uint8_t inverted)
{
	uint8_t w = ksdF8StringWidth(s);
	ksdF8WriteString(s,X-w/2,Y,inverted);
}

uint8_t ksdF8StringWidth(char* s) __attribute__((section(".highmem")));
uint8_t ksdF8StringWidth(char* s)
{
	uint8_t nc = strlen(s);
  uint16_t width = 0;
	for (uint8_t i=0;i<nc;i++)
	{
    width += ksdLetterSpacing + pgm_read_byte(&lentbl_S[s[i]-32]);
	}
	return width;
}

// Font16 - 16 bit word vertical bytes
// LSbit at the top. Characters use bits 2 - 15 - so shift up 2 bits
// so *ppY % 8 == 0 or 1 then character will only affect two vertical bytes
void ksdF16WriteChar(uint8_t ch, uint16_t* ppX, uint16_t* ppY) __attribute__((section(".highmem")));
void ksdF16WriteChar(uint8_t ch, uint16_t* ppX, uint16_t* ppY)
{
	uint8_t x, cbits = 0;
	uint8_t a, b, Y;
	uint8_t mask, cbyte;
	uint16_t mword;
	uint16_t cword[12];
	uint8_t bits[12];
	uint16_t * pDigitWords = (uint16_t*)pgm_read_word(&chrtbl_L[ch-32]);
	uint8_t chlen = pgm_read_byte(&lentbl_L[ch-32]) + ksdLetterSpacing;
	Y = *ppY / 8; // upper byte y
	a = *ppY % 8;
	b = 8 - a;
 // top byte
	mask = (0xFF >> b);		
	ksdGoTo(*ppX, Y, 1);
	for (x = 0; x < chlen; x++)
	{
		if (screen_x == 64){ksdWriteCommand(KS_YSET | 0, 1); ksdReadData();}
		if (x < (chlen-ksdLetterSpacing))
		{
			cword[x] = pgm_read_word(&pDigitWords[x]);
			cword[x] >>= 2;  // shift up two bits
		}
		else
		{cword[x] = 0;} // the letter spcaing also need read-modify-write
		cbyte = cword[x] % 0x100;
		cbits = (cbyte << a);
		bits[x] = (ksdReadData() & mask) | cbits;
	}
	ksdGoTo(*ppX, Y, 0);
	for (x = 0; x < chlen; x++)
	{ 
	 	ksdWriteData(bits[x]);
	}
// middle byte
	if (a < 3)
	{
		mask = 0xC0 << a; 
		ksdGoTo(*ppX, Y+1, 1);
	}
	for (x = 0; x < chlen; x++)
	{
		mword = (cword[x] >> b);
		if (a < 3)
		{
			cbyte = (mword % 100) &(!mask);
			bits[x] = (ksdReadData() & mask) | cbits;
		}
		else
		{
			bits[x] = mword % 0x100;
		} 		
	}	
	ksdGoTo(*ppX, Y+1, 0);
	for (x = 0; x < chlen; x++)
	{ 
		mword = cword[x] >> b; 
		ksdWriteData(mword % 0x100);
	}
// bottom byte
	if (a >= 3)
	{
		mask = (0xFF << (a-2));		
		ksdGoTo(*ppX, Y+2, 1);
		for (x = 0; x < chlen; x++)
		{
			cbyte = cword[x] / 0x100;
			cbits = (cbyte >> b);
			bits[x] = (ksdReadData() & mask) | cbits;
		}
		ksdGoTo(*ppX, Y+2, 0);
		for (x = 0; x < chlen; x++)
		{ 
		 	ksdWriteData(bits[x]);
		}
	}
	*ppX += chlen;  // position for next (proportional spacing)

}


void ksdF16WriteString(char* s, uint16_t X, uint16_t Y) __attribute__((section(".highmem")));
void ksdF16WriteString(char* s, uint16_t X, uint16_t Y)
{
	uint16_t StartX = X;
	uint16_t nc = strlen(s);
	char ch;
	for (uint16_t i = 0; i < nc; i++)
	{
		ch = s[i];
		if (ch != 10)
		{
			if (ch == 13)
			{
				Y += ksdLineSpacing;
				X = StartX;
			}
			else {ksdF16WriteChar(s[i], &X, &Y);}
		}
	}
	ksdNextCharX = X;
}


void ksdWriteFreqAString(char* s, uint16_t X, uint16_t Y) __attribute__((section(".highmem")));
void ksdWriteFreqAString(char* s, uint16_t X, uint16_t Y)
{
	uint16_t i, nc;
	char ch;

	nc = strlen(s);
	for (i = 0; i < nc; i++) {
		ch = s[i];
		switch (ch) {
			case ' ': ch = 148; break;
			case '+': ch = 140; break;
			case '-': ch = 141; break;
			case '.': ch = 142; break;
			case 'A': ch = 143; break;
			case 'B': ch = 144; break;
			case 'R': ch = 145; break;
			case 'T': ch = 146; break;
			case 'G': ch = 147; break;
			default: ch += 82;
		}
		ksdF16WriteChar(ch, &X, &Y);
	}
}


void ksdWriteFreqBString(char* s, uint16_t X, uint16_t Y) __attribute__((section(".highmem")));
void ksdWriteFreqBString(char* s, uint16_t X, uint16_t Y)
{
	uint16_t i, nc;
	char ch;

	nc = strlen(s);
	for (i = 0; i < nc; i++) {
		ch = s[i];
		switch (ch) {
			case ' ': ch = 154;  break;
			case '+': ch = 146; break;
			case '-': ch = 147; break;
			case '.': ch = 148; break;
			case 'A': ch = 149; break;
			case 'B': ch = 150; break;
			case 'R': ch = 151; break;
			case 'T': ch = 152; break;
			case 'G': ch = 153; break;
			default: ch += 88;
		}
		ksdF8WriteChar(ch, &X, &Y, 0);
	}
}


void ksdWrite5x3String(char* s, uint16_t X, uint16_t Y) __attribute__((section(".highmem")));
void ksdWrite5x3String(char* s, uint16_t X, uint16_t Y)
{
	uint16_t nc = strlen(s);
	char ch;
	for (uint16_t i = 0; i < nc; i++)
	{
		ch = s[i];
		ksdF8WriteChar(f8Map5x3(ch), &X, &Y,0);		
	}
}
