/*
 *****    homebrew-radios.net   *****

 *****     Project TrxAVR       *****


 taFT245R.c    program file
 
Support for FTDI FT245RL USB chip 
 
**********
See notes on usbWriteBlock for info on use of pointers and typecasts
**********

Note that USB is sharing PORTC with display and key pad
- so must set FT245RL back to write mode so its data inputs
  will be high impedance

Interrupts provided for but not actually used

Timeout on read of write to be done in external driving code 
because timeout actions will differ for different uses of USB
*/


#include "taGlobal.h"
#include "taFT245R.h"
#include "taCharLCD.h"
#include "taDebug.h"
#include "ta24LC512.h"
#include "taFT245R.h"
#include "taStarDSP.h"
#include "taSWR.h"
#include "taConfig.h"
#include "taDDS.h"
#include "taDiagnostic.h"
#include "taI2C.h"
#include "taStarControl.h"
#include "taRTC.H"




#define portUSBdata PORTC
#define pinUSBdata  PINC
#define portUSBctrl PORTD
#define pinUSBctrl  PIND
#define usbRXF   3          // also on INT3
#define usbTXE   2    		// also on INT2
#define usbRD    5
#define usbPWREN 6
#define usbWR    4

#define LFchar  10
#define CRchar  13

#define INT2sense 2;   //Interrupt on falling edge
#define INT3sense 2;   //Interrupt on falling edge


unsigned char RxIntEnabled;
char ReqString[16];
char StrUsbA[80];   
char StrUsbB[80];   
char StrUsbC[8];   
static uint8_t DollarCount;
static unsigned char ReqFlag;

void usbDoCheckPC();


// we are not using interrput on outgoing characters
ISR(INT2_vect)   // interrupt on TXE line
{
  // code here
  return;
}


// USB receive interrupt used only for "$$$" at start of request string
ISR(INT3_vect)   // interrupt on RXE line
{
	unsigned char c;
	if (usbGetByte(&c) == 0) return;
	if((DollarCount > 0) && (c != '$')) {usbResetRx(); return;}
	if (c == '$')
	{
		DollarCount += 1;
		if (DollarCount == 3)
		{
      		DollarCount = 0;
			ReqFlag = 1;
			usbINTenable(3,0); // disable Rx interrupt INT3
			RxIntEnabled = 0;
			if (usbReceiveString(ReqString,10) == 0){usbResetRx();}  // 10ms timeout
		}
	}
	else usbResetRx();
	return;
}


void usbResetRx()
{
	DollarCount = 0;
	ReqFlag = 0;
	RxIntEnabled = 1;
	usbINTenable(3,1); // enable Rx interrupt INT3
	return;
}


void usbInit()
{
	unsigned char s2 = INT2sense;
	unsigned char s3 = INT3sense;
	
	portUSBctrl |= (1 << usbRD); // set RD high (RD low will stop display and keypad working)
	portUSBctrl &= ~(1 << usbWR);  // set WR low
	EICRA &= 0x0F;    // clear INT2 and INT3 sense bits
	EICRA |= ((s3 << 6) | (s2 << 4));  // set INT3 and INT2 sense bits 
	usbINTenable(2,0); // disable Tx interrupt INT2
	usbResetRx();
	return;
}


void usbINTenable(unsigned char n, unsigned char tf)
{
	if (tf == 0)
		EIMSK &= ~(1 << n);
	else
		EIMSK |= (1 << n);
	return;
}


// Checks state of RXF# 
// If low then byte available - reads it and returns 1
// If high returns 0
int usbGetByte(uint8_t * c)
{
	unsigned char b = 0;  

	if ((pinUSBctrl & (1 << usbRXF)) != 0) return 0; //no read if RXF# high
 	DDRC = 0; // port C set to input
 	PORTC = 0; // port C low - Hi Z
	portUSBctrl &= ~(1 << usbRD);  // set RD# low - FIFO data onto PORTC bus
	// 185ns delay for FT245 data assertion and AVR input synchronising latch
	__asm__ volatile ("nop \r\n nop \r\n nop \r\n" : : );
	b = pinUSBdata;
	portUSBctrl |= (1 << usbRD);  // set RD# high - 
	*c = b;
	return 1;
}


// Checks state of TXE# 
// If low then byte can be written - writes it and returns 1
// If high returns 0
int usbPutByte(unsigned char b)
{
	if ((pinUSBctrl & (1 << usbTXE)) != 0) return 0;   // no send if TXE# high
	DDRC = 0xFF;   // PORTC to output
	portUSBdata = b;               // data on bus
	portUSBctrl |= (1 << usbWR);   // WR high
	// 62.5ns delay
	__asm__ volatile ("nop \r\n" : : );
	portUSBctrl &= ~(1 << usbWR);   // WR low  - falling edge clocks data into FT245RL
	return 1; 
}


uint8_t usbReceiveBlock(uint8_t *start, uint16_t noofbytes, uint16_t timelimit)
{
	uint8_t *p;
	uint16_t fails;
	for (p = start; p < (start + noofbytes); p++)
	//timelimit = (timelimit/3);
	{
		fails = 0;
		do
		{
			if (usbGetByte(p) == 0)  // attempts to read a byte - returns 0 if fails
			{
				fails += 1;   // increment failure counter 
				_delay_ms(1);  
				if (fails > timelimit) return 0;  // exit returning 0 it timeout
			}
			else
			{
				fails = 0;   //charater received -   reset failure counter to zero
			}	
		} while (fails > 0);
	}
	return 1;
}		
	
	
//	usbSendBlock  - sends noofbytes bytes. Parameter 'start' is a pointer to the
//	firt byte of the block. The block is send into variable(s) set up by the calling
//	routine prior to calling usb SendBlock, eg:
//			unit8_t block[12]   // an array of 12 bytes  (using uint8_t  = unsigned 8 bit)
//			uint8_t *pb;  	// define a typed pointer pb which can point to a byte variable
												// of type uint8_t. 
//			usbSendBlock(pb,8,1000); // send the block 1000mS timeout
//	Thus WE (the calling software, reserve the storage and give the usbSendBlock 
//	functoin a pointer to the storage so thatusb SendBlock can read the usb bytes into
//	OUR storage.  This means that usbSendBlock can handle any size of block
//	as it doesn't have to allocate any internal storage of its own.
//
//	Typecasts: note that because the strict typing of C variables applies also to
//	pointers to variables, we will usually have to use a typecast if the calling
//	routine's storage is not of type uint8_t.
//			uint8_t *pb;	// pb is ponter to variables of type uint8_t
//			long fred;		// our data for transmission is in a four-byte signed variable fred.
//			fred = 35*X/y; // etc - ie assign a value to fred
//			pb = (*uint8_t)fred;  // pb points to the first byte of fred .. 
//			but keeps usbSendByte happy by being s pointer to type 'uint8_t' rather
//			than a pointer to type 'long'.
//
uint8_t usbSendBlock(uint8_t *start, uint16_t noofbytes, uint16_t timelimit)
{
	uint8_t *p;
	uint16_t fails;
	for (p = start; p < (start + noofbytes); p++)
	{
		fails = 0;
		do
		{
			if (usbPutByte(*p) == 0)  // attempts to send a byte - returns 0 if fails
			{
				fails += 1;   // increment failure counter 
				_delay_ms(1);  
				if (fails > timelimit) return 0;  // exit returning 0 it timeout
			}
			else
			{
				fails = 0;   //charater received -   reset failure counter to zero
			}	
		} while (fails > 0);
	}
	return 1;
}		
	

// function to write a string
// if any character fails within timelimit mS the function returns 0
// successful send returns 1
// s is a pointer to the start of a character array set up by the calling code
// s must contain a null-terminated string
// if terminator = 0:  send terminating 0
// if terminator = 1:  don't send terminating 0
// if terminator = 13   send  ascii 13 and 10 on the end
//     ***  source buffer must have space for 13 and 10 ***
// (any other valus of terminator will terminate with 0.
uint8_t usbSendString(char *s, uint16_t timelimit, uint8_t terminator)
{
	uint16_t nc = strlen(s);  // strlen returns string length in nc (string.h)
	uint16_t fails;  // counts number of failed attemps on a 1ms delay loop
	switch (terminator)
	{
		case 0:  break;
		case 1:  nc -= 1; break;
		case 13: s[nc++] = 13; s[nc] = 10; break;
		default: return 0;
	}
	for (uint16_t i = 0; i <= nc; i++)  // for i = 0 to nc  (Send the terminating zero) 
	{
		fails = 0;
		do
		{
			if (usbPutByte(s[i]) == 0)  // attempts to write a char - returns 0 if fails
			{
				fails += 1;   // increment failure counter 
				_delay_ms(1);  
				if (fails > timelimit) return 0;  // exit returning 0 it timeout
			}
			else fails = 0;   //charater sent -   reset failure counter to zero
		}
		while (fails > 0);  // repeat the do .. while loop if fails > 0
	}
	return 1;   // string sent     return
}


// function to read a string
// if a character read fails within timelimit mS the function returns 0
// successful read returns 1
// s is a pointer to the start of a character array set up by the calling code
// after a successful read, s will contain a null-terminated string
// The incoming string must be terminated by:
//  a zero byte, OR by cr followed by lf, OR by lf followewd by cr
// Once a cr has been received, the next char must be an lf   = otherwise failure
// Once a lf has been received, the next char must be an cr   = otherwise failure

uint8_t usbReceiveString(char *s, uint16_t timelimit)
{
	uint16_t nc;  // nc counts number characters
	uint16_t fails;  // counts number of failed attemps on a 1ms delay loop
	unsigned char ch;
	uint8_t CR_received = 0;
	uint8_t LF_received = 0;
	uint8_t done = 0;
	nc = 0;
	do
	{
		fails = 0;
		do
		{
			if (usbGetByte(&ch) == 0)  // attempts to read a char - returns 0 if fails
			{
  
				fails += 1;   // increment failure counter 
				_delay_ms(1);  
				if (fails > timelimit) return 0;  // exit returning 0 it timeout
			}
			else
			{
 				fails = 0;   //character received -   reset failure counter to zero
				if (CR_received > 0) 
				{
					if (ch == LFchar) {done = 1; break;} else return 0;
				}
				if (LF_received > 0)
				{
					if (ch == CRchar) {done = 1; break;} else return 0;
				}
				if (ch == LFchar) {LF_received = 1; break;};
				if (ch == CRchar) {CR_received = 1; break;};
				if (ch == 0) { done = 1; break;}; 
				s[nc++] = ch;
			}	
		}
		while (fails > 0);  // repeat the do .. while loop if fails > 0
 	}
	while (done == 0);
	s[nc] = 0;
	return 1;   // string received     return
}

/////////////////////////////////////////////////////////////////////////////

///////////////  PC command  handling ///////////////////////////////////////

//  Commands of form '$$$command '   - not the terminating space
//  Interrupt vector filters out the $$$ 
// and then puts the 'command' in ReqString and sets ReqFlag = 1
// usbCheckPC is polled by the main loop

// Stacks - load and save in taDDS.c module
// cDDSclock, offsets, encoders8 assignments  - load and save in taConifg.c module

void usbCheckPC()
{
	do
	{
		usbDoCheckPC();
		if(DiagMode!='X'){_delay_ms(200);}
	}
	while(DiagMode!='X');
}


void usbDoCheckPC()
{
	if (ReqFlag == 0){return;}
	ReqFlag = 0;
  

	// Control commands - sub decode on final character
	if (strncmp(ReqString,"CTL_",4)==0)     // eg: $$$CTL_T. 'A' is char 4 of "CTL_T"
	{
		scUsbRequest(ReqString[4]);
		RxIntEnabled = 1;
		usbINTenable(3,1); // enable Rx interrupt INT3
		return;
	}
	// DSP commands - sub decode on final character
	if (strncmp(ReqString,"DSP_",4)==0)     // eg: $$$DSP_A. 'A' is char 4 of "DSP_A"
	{
		dspUsbRequest(ReqString[4]);
		RxIntEnabled = 1;
		usbINTenable(3,1); // enable Rx interrupt INT3
		return;
	}

	// DDS commands - sub decode on final character
	if (strncmp(ReqString,"DDS_",4)==0)  // eg: $$$DSP_A. 'A' is char 4 of "DSP_A"
	{
		ddsUsbRequest(ReqString[4]);
		RxIntEnabled = 1;
		usbINTenable(3,1); // enable Rx interrupt INT3
		return;
	}  

	// SWR meter commands - sub decode on final character
	if (strncmp(ReqString,"SWR_",4)==0)    // eg: $$$SWR_A. 'A' is char 4 of "SWR_A"
	{
		swUsbRequest(ReqString[4]);
		RxIntEnabled = 1;
		usbINTenable(3,1); // enable Rx interrupt INT3
		return;
	}

	// DEBUG  commands - sub decode on final character
	if (strncmp(ReqString,"DBG_",4)==0)
	{
		dbUsbRequest(ReqString[4]);   // eg: $$$DBG_V. 'V' is char 4 of "SWR_V"
		RxIntEnabled = 1;
		usbINTenable(3,1); // enable Rx interrupt INT3
		return;
	} 

	// CONFIG  commands - sub decode on final character
	if (strncmp(ReqString,"CFG_",4)==0)   // eg: $$$CFG_U. 'U' is char 4 of "CFG_U"
	{
		cfUsbRequest(ReqString[4]);
		RxIntEnabled = 1;
		usbINTenable(3,1); // enable Rx interrupt INT3
		return;
	} 

	// EEMEM  Stack Load commands - sub decode on final character
	// eg: $$$L_STK_S. 'S' is char 6 of "L_STK_S"  (S = Band SSB Stack)
	if (strncmp(ReqString,"L_STK_",6)==0)
	{
		ddsStackLoadRequest(ReqString[6]);
		RxIntEnabled = 1;
		usbINTenable(3,1); // enable Rx interrupt INT3
		return;
	}

	// EEMEM  Band Stack Save commands - sub decode on final character
	// eg: $$$S_BND_7. '7' is char 6 of "S_STK_7"   (7 = Mem Stack 67 )
	if (strncmp(ReqString,"S_STK_",6)==0)
	{
		ddsStackSaveRequest(ReqString[6]);
		RxIntEnabled = 1;
		usbINTenable(3,1); // enable Rx interrupt INT3
		return;
	}

	// EEMEM  I2C config. Load commands - sub decode on final character
	// eg: $$$L_I2C_D. 'D' is char 6 of "L_STK_S" 
	if (strncmp(ReqString,"L_I2C_",6)==0)
	{
		icI2CLoadRequest(ReqString[6]);
		RxIntEnabled = 1;
		usbINTenable(3,1); // enable Rx interrupt INT3
		return;
	}


	// EEMEM  I2C config Save commands - sub decode on final character
	// eg: $$$S_I2C_D. 'D' is char 6 of "S_I2C_D"  
	if (strncmp(ReqString,"S_I2C_",6)==0)
	{
		icI2CSaveRequest(ReqString[6]);
		RxIntEnabled = 1;
		usbINTenable(3,1); // enable Rx interrupt INT3
		return;
	}

	// RTC  set and read - sub decode on final character
	// eg: $$$RTC_S. 'S' is char 4 of "RTC_S"  
	if (strncmp(ReqString,"RTC_",4)==0)
	{
		rtcUsbRequest(ReqString[4]);
		RxIntEnabled = 1;
		usbINTenable(3,1); // enable Rx interrupt INT3
		return;
	}



	
	// Diagnostics - sub decode on final character
	if (strncmp(ReqString,"DIA_",4)==0)     // eg: $$$DIA_P. 'p' is char 4 of "DIA_P"
	{
		diaUsbRequest(ReqString[4]);
		RxIntEnabled = 1;
		usbINTenable(3,1); // enable Rx interrupt INT3
		return;
	}
}	 
