/*
 *****    homebrew-radios.net   *****

 *****     Project TxrAVR       *****


 ta24LC512.c    program file
 
Functions using the 24LC512 64kbyte EEPROM chip on two wire interface (TWI)  
*/

#include "taGlobal.h"
#include "ta24LC512.h"
#include "taTWI.h"
#include "taFT245R.h"
#include "taStarDSP.h"
#include "taStarControl.h"
#include "taDDS.h"

struct CalDataStruct_type CalDataStruct;
uint8_t xeebuf[132];

/////// External EEPROM SWR calibration information ////

// This data is transferred to RAM variables  - defined in ta24LC512.h//

	// Hardware settings  - 96 bytes
static const uint16_t XEEadHardwareSettings = 0x7580;

static const uint16_t XEEtftaI2Caddress = 0x7600;

	// 7 sticky VFO slots  - address of stored pointer to oldest  - one byte
static const uint16_t XEEadOldestStickyVfoSlotNo = 0x765E;

	// Reflected power calibration points 0 to 31
	// = ADC output levels 0 32, 64 ....992
static const uint16_t XEEadRefCalLevels = 0x7680;  // start of 64 bytes = 32 levels

	// Forward power calibration points 0 to 31
	// = ADC output levels 0 32, 64 ....992
static const uint16_t XEEadFwdCalLevels = 0x76C0;  // start of 64 bytes = 32 levels

	// Mismatch seconds
static const uint16_t XEEadMismatchSeconds = 0x77F8;

	// Mismatch deciwatts
static const uint16_t XEEadMismatchDeciwatts = 0x77FA;

	// Reflected analogue full scale watts
static const uint16_t XEEadRefFsWatts = 0x77FC;

	// Forward analogue full scale watts
static const uint16_t XEEadFwdFsWatts = 0x77FE;

 	
//////// External EEPROM DSP parameter storage /////////////////

	// External 24LC512 EEPROM parameter addresses in load sequence 0 to 8
	// then backup copies 9 to 14
	//  used for loading from PC only 
static const uint16_t dspParamsAddressList[10] = {
		0x7F80,   //  0  green   SSB
		0x7B80,   //  1  yellow  SSB
		0x7D80,   //  2  red     SSB
		0x7880,   //  3  max value
		0x7980,   //  4  min value
		0x7E80,   //  5  green   CW
		0x7A80,   //  6  yellow  CW
		0x7C80,   //  7  red     CW
		0x7900,   //  8  user ID
		0x7800};  //  9  multiplier  (not transmitted in picastar) 
		    
	// Load routine subtracts 0x80 for the backup copies

//		0x7F00,   //  9  green   SSB  backup
//		0x7B00,   // 10  yellow  SSB  backup
//		0x7D00,   // 11  red     SSB  backup
//		0x7E00,   // 12  green   CW   backup
//		0x7A00,   // 13  yellow  CW   backup
//		0x7C00};  // 14  red     CW   backup  

// Lots of 28 byte spaces free above each 100byte parameter block

//  Block at 0x7FE8
static const uint16_t XEEadByteCount = 0x7FE8; //  2 bytes   = DSP code byte count   

// IJS 26.08.2012    Tx Drive now stored per band.mode.colour as 10 bytes starting
// at colour.mode params address (above) + 0x70.  
// ie 128 bytes for each colour.mode.  First 99 bytes for params.  Top 16bytes for Tx drive


// Local function definitions
void xeWordsToBuf(uint16_t* ram, uint8_t n);
void xeBufToWords(uint16_t* ram, uint8_t n);


///////  SWR calibration load and save ////

void xeWordsToBuf(uint16_t* ram, uint8_t n)
{
	uint8_t i;
	
	for (i = 0; i < 2*n; i++)
	{
		xeebuf[i] = *((uint8_t*)ram + i);  
	}
}

void xeBufToWords(uint16_t* ram, uint8_t n)
{
	uint8_t i;

	for (i = 0; i < 2*n; i++)
	{
		*((uint8_t*)ram + i) = xeebuf[i];  
	}
}


void xeLoadSwrCalData()
{
	xeeprom_blockread(XEEadFwdCalLevels,64);
	xeBufToWords(CalDataStruct.cdsFwdCalLevel,32);
	xeeprom_blockread(XEEadRefCalLevels,64);
	xeBufToWords(CalDataStruct.cdsRefCalLevel,32);

	xeeprom_blockread(XEEadFwdFsWatts,2);
	xeBufToWords(&CalDataStruct.cdsFwdFsWatts,1);
	xeeprom_blockread(XEEadRefFsWatts,2);
	xeBufToWords(&CalDataStruct.cdsRefFsWatts,1);

	xeeprom_blockread(XEEadMismatchDeciwatts,2);
	xeBufToWords(&CalDataStruct.cdsMismatchDeciWatts,1);
	xeeprom_blockread(XEEadMismatchSeconds,2);
	xeBufToWords(&CalDataStruct.cdsMismatchSeconds,1);
}


void xeSaveSwrCalData()
{
	xeWordsToBuf(CalDataStruct.cdsFwdCalLevel,32);
	xeeprom_blockwrite(XEEadFwdCalLevels,64);
	xeWordsToBuf(CalDataStruct.cdsRefCalLevel,32);
	xeeprom_blockwrite(XEEadRefCalLevels,64);

	xeWordsToBuf(&CalDataStruct.cdsFwdFsWatts,1);
	xeeprom_blockwrite(XEEadFwdFsWatts,2);
	xeWordsToBuf(&CalDataStruct.cdsRefFsWatts,1);
	xeeprom_blockwrite(XEEadRefFsWatts,2);

	xeWordsToBuf(&CalDataStruct.cdsMismatchDeciWatts,1);
	xeeprom_blockwrite(XEEadMismatchDeciwatts,2);
	xeWordsToBuf(&CalDataStruct.cdsMismatchSeconds,1);
	xeeprom_blockwrite(XEEadMismatchSeconds,2);
}

//TftA I2C address

void xeSaveTftaI2Caddress()
{
  xeWriteByte(XEEtftaI2Caddress, TftaI2Caddress);
}


void xeLoadTftaI2Caddress()
{
  xeReadByte(XEEtftaI2Caddress, &TftaI2Caddress);
}



//  Hardware setting
void xeSaveHardwareSettings()
{  
	memcpy(&xeebuf[0],&HardwareSettings.hsDisplay,96);
	xeeprom_blockwrite(XEEadHardwareSettings,96);
}


void xeLoadHardwareSettings()
{
	xeeprom_blockread(XEEadHardwareSettings,96);
	memcpy(&HardwareSettings.hsDisplay,&xeebuf[0],96);
}


// variable n depands on mode and colour 
uint16_t xeGetDspParamsAddress(uint8_t n)
{
	return dspParamsAddressList[n];
}


uint16_t xeGetDspMaxAddr()
{
	return dspParamsAddressList[3];
}


uint16_t xeGetDspMinAddr()
{
	return dspParamsAddressList[4];
}


uint16_t xeGetDspMultAddr()
{
	return dspParamsAddressList[9];
}




uint16_t xeGetOldestStickyVfoSlotNoAddr()
{
	return XEEadOldestStickyVfoSlotNo;
}


uint16_t xeGetDspCodeByteCount()
{
	uint8_t b0;
	uint8_t b1;
	
	xeReadByte(XEEadByteCount, &b0);   //store location 
	xeReadByte(XEEadByteCount+1, &b1); // (store location + 1) i.e 0x0000 2 byte figure KGQ
	return 256*b0 + b1 - 14;  // deduct 14 for the received  #############x
}


void xeSetTxDriveToDefault()
{
	// set Tx drive levels from green params
	uint8_t b;
	uint8_t pcol;
	uint8_t drive = dspGetGreenParamValue(81);  // Tx drive
	
	for(pcol=pgreen;pcol<=pred;pcol++)  // pgreen =1   pyellow = 2   pred = 3
	{
		for(b=0;b<10;b++)   // band
		{
			xeWriteDrive(drive,b,0,pcol);  // SSB
			xeWriteDrive(drive,b,1,pcol);  // CW
		}
	}
}


void xeWriteDrive(uint8_t d, uint8_t band, uint8_t mode, uint8_t pcolour)   // mode = 0 ro SSB and 1 for cw
{
	uint16_t ParamsAddress;
	ParamsAddress = xeGetDspParamsAddress((5 * (Mode / 2)) + pcolour - 1);
	xeWriteByte(ParamsAddress + 0x70 + Band, d);
}


uint8_t xeReadDrive(uint8_t band, uint8_t mode, uint8_t pcolour)   // mode = 0 ro SSB and 1 for cw
{
	uint16_t ParamsAddress;
	uint8_t d = 0;
	ParamsAddress = xeGetDspParamsAddress((5 * (Mode / 2)) + pcolour - 1);
	xeReadByte(ParamsAddress + 0x70 + Band, &d);
	return d;
}


// returns 1 if ok, 0 if not
int xeWrite32bit(uint16_t eeaddr, uint32_t n)
{
	unsigned char b;
	for (int i=0;i<4;i++)
	{
		b = n % 0x100; 
		if (xeeprom_write_byte(eeaddr+i, b) != 1) return 0;
		n /= 0x100;
	}
	return 1;
}


// returns 1 if ok, 0 if not
int xeRead32bit(uint16_t eeaddr, uint32_t *n)
{ 
	char b;
	uint32_t nu = 0;
	for (int i=3;i>=0;i--)
	{
		if (xeeprom_read_byte(eeaddr+i, &b) != 1) return 0;
		nu += b;
		if (i > 0) {nu *= 0x100;};     
	}
	*n = nu;
	return 1;
}	


// returns 1 if ok, 0 if not
int xeWriteByte(uint16_t eeaddr, uint8_t n)
{
	unsigned char b = n;
	if (xeeprom_write_byte(eeaddr, b) != 1) return 0;
	return 1;
}


// returns 1 if ok, 0 if not
int xeReadByte(uint16_t eeaddr, uint8_t *n)
{ 
	 char b;   
	if (xeeprom_read_byte(eeaddr, &b) != 1) return 0;
	*n = b;
	return 1;//if 1 location = FF
}	
